#!/usr/bin/env python3
"""
Graph converter for Resource-Constrained Modulo Scheduling Problem (RCMSP).
Created using subagent_prompt.md version: v_02

This problem is about cyclic resource-constrained project scheduling with 
generalised precedence relations. Tasks are repeated infinitely with the same period,
constrained by scarce cumulative resources and precedence constraints with latencies/distances.

Key challenges: 
- Cyclic scheduling with precedence constraints spanning iterations
- Resource capacity constraints across multiple resources
- Minimizing both period and makespan
- Complex precedence relations with latencies and distances
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the RCMSP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with explicit constraint nodes
    - Task nodes (type 0): Decision variables with resource requirements
    - Precedence constraint nodes (type 1): Individual precedence relations
    - Resource constraint nodes (type 1): Cumulative resource capacity constraints
    - Edges model task participation in constraints with resource/precedence weights
    """
    # Extract data
    n_res = json_data.get('n_res', 0)
    n_tasks = json_data.get('n_tasks', 0)
    n_prec = json_data.get('n_prec', 0)
    rcap = json_data.get('rcap', [])
    
    # Reshape flattened arrays back to 2D
    rreq_flat = json_data.get('rreq', [])
    prec_flat = json_data.get('prec', [])
    
    # Reshape rreq: n_tasks × n_res
    rreq = []
    for t in range(n_tasks):
        row = []
        for r in range(n_res):
            idx = t * n_res + r
            if idx < len(rreq_flat):
                row.append(rreq_flat[idx])
            else:
                row.append(0)
        rreq.append(row)
    
    # Reshape prec: n_prec × 4 (task1, task2, latency, distance)
    prec = []
    for p in range(n_prec):
        row = []
        for c in range(4):
            idx = p * 4 + c
            if idx < len(prec_flat):
                row.append(prec_flat[idx])
            else:
                row.append(0)
        prec.append(row)
    
    G = nx.Graph()
    
    # Task nodes with criticality-based weights
    max_total_req = 1
    for t in range(n_tasks):
        # Calculate total resource requirement across all resources
        total_req = sum(rreq[t][r] for r in range(n_res) if r < len(rreq[t]))
        max_total_req = max(max_total_req, total_req)
    
    for t in range(n_tasks):
        total_req = sum(rreq[t][r] for r in range(n_res) if r < len(rreq[t]))
        
        # Weight by resource criticality (higher req = more critical)
        criticality = total_req / max_total_req if max_total_req > 0 else 0.5
        
        # Special handling for source/sink tasks (tasks 1 and n_tasks have 0 duration)
        if t == 0 or t == n_tasks - 1:  # 0-indexed, so task 1 is index 0, task n_tasks is index n_tasks-1
            weight = 0.9  # High importance as schedule boundaries
        else:
            weight = max(0.1, criticality)
        
        G.add_node(f'task_{t+1}', type=0, weight=weight)
    
    # Precedence constraint nodes with latency-based weights
    max_latency = max((prec[p][2] for p in range(n_prec) if len(prec[p]) > 2), default=1)
    
    for p in range(n_prec):
        if len(prec[p]) >= 4:
            task1, task2, latency, distance = prec[p][:4]
            
            # Weight by constraint tightness (higher latency = tighter timing constraint)
            if max_latency > 0:
                tightness = (latency + 1) / (max_latency + 1)  # +1 to handle zero latencies
            else:
                tightness = 0.5
            
            # Boost weight for cross-iteration precedences (distance > 0)
            if distance > 0:
                tightness = min(1.0, tightness * 1.5)  # Cyclic constraints are more complex
            
            G.add_node(f'prec_{p}', type=1, weight=tightness)
    
    # Resource constraint nodes with utilization-based weights
    for r in range(n_res):
        capacity = rcap[r] if r < len(rcap) else 1
        
        # Calculate total demand for this resource
        total_demand = sum(rreq[t][r] for t in range(n_tasks) if r < len(rreq[t]))
        
        # Weight by resource tightness
        if capacity > 0:
            utilization = total_demand / capacity
            # Use exponential scaling for utilization to emphasize bottlenecks
            tightness = min(1.0, math.exp(utilization - 1.0))  # exp(0) = 1.0 for 100% utilization
        else:
            tightness = 1.0  # Zero capacity = maximum tightness
        
        G.add_node(f'resource_{r}', type=1, weight=tightness)
    
    # Bipartite edges: tasks to precedence constraints
    for p in range(n_prec):
        if len(prec[p]) >= 4:
            task1, task2, latency, distance = prec[p][:4]
            
            # Edge weight based on constraint participation strength
            participation_weight = 0.8 if distance > 0 else 0.6  # Cyclic constraints more important
            
            # Connect both tasks involved in the precedence
            G.add_edge(f'task_{task1}', f'prec_{p}', weight=participation_weight)
            G.add_edge(f'task_{task2}', f'prec_{p}', weight=participation_weight)
    
    # Bipartite edges: tasks to resource constraints
    for t in range(n_tasks):
        for r in range(n_res):
            if r < len(rreq[t]) and rreq[t][r] > 0:
                capacity = rcap[r] if r < len(rcap) else 1
                
                # Weight by resource consumption ratio
                consumption_ratio = rreq[t][r] / capacity if capacity > 0 else 1.0
                
                # Use square root to emphasize high consumers but keep weights reasonable
                edge_weight = min(1.0, math.sqrt(consumption_ratio))
                
                G.add_edge(f'task_{t+1}', f'resource_{r}', weight=edge_weight)
    
    # Add conflict edges between tasks competing for highly utilized resources
    for r in range(n_res):
        capacity = rcap[r] if r < len(rcap) else 1
        
        # Find tasks using this resource
        resource_users = []
        for t in range(n_tasks):
            if r < len(rreq[t]) and rreq[t][r] > 0:
                resource_users.append((t, rreq[t][r]))
        
        # Calculate total demand
        total_demand = sum(req for _, req in resource_users)
        
        # Add conflict edges for oversubscribed resources
        if total_demand > capacity * 1.2:  # 20% oversubscription threshold
            # Sort by demand (highest first)
            resource_users.sort(key=lambda x: x[1], reverse=True)
            
            # Add conflicts between top consumers
            for i in range(min(len(resource_users), 4)):  # Top 4 consumers
                for j in range(i + 1, min(len(resource_users), 4)):
                    t1, req1 = resource_users[i]
                    t2, req2 = resource_users[j]
                    
                    # Only add conflict if they can't both run at full capacity
                    if req1 + req2 > capacity:
                        conflict_strength = (req1 + req2) / (2 * capacity)
                        conflict_strength = min(1.0, conflict_strength)
                        
                        G.add_edge(f'task_{t1+1}', f'task_{t2+1}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()