#!/usr/bin/env python3
"""
Graph converter for Resource-Constrained Project Scheduling Problem (RCPSP).
Created using subagent_prompt.md version: v_02

This problem is about scheduling tasks with precedence constraints and resource limitations.
Key challenges: 
- Resource contention (multiple tasks competing for limited resources)
- Precedence dependencies creating critical paths
- Makespan optimization requiring careful scheduling
- Resource utilization patterns affecting schedule feasibility
"""

import sys
import json
import math
import re
import networkx as nx
from pathlib import Path


def parse_dzn_successors(dzn_file):
    """Parse successor information from DZN file since JSON conversion doesn't handle sets."""
    successors = {}
    
    with open(dzn_file, 'r') as f:
        content = f.read()
    
    # Find the suc array definition
    suc_match = re.search(r'suc\s*=\s*\[(.*?)\];', content, re.DOTALL)
    if not suc_match:
        return successors
    
    suc_content = suc_match.group(1)
    
    # Parse each task's successors
    task_patterns = re.findall(r'\{\s*([^}]*)\s*\}', suc_content)
    
    for task_idx, pattern in enumerate(task_patterns):
        task_id = task_idx + 1  # 1-indexed in MiniZinc
        if pattern.strip():
            # Parse comma-separated successor IDs
            succ_list = [int(x.strip()) for x in pattern.split(',') if x.strip()]
            successors[task_id] = succ_list
        else:
            successors[task_id] = []
    
    return successors


def build_graph(mzn_file, json_data):
    """
    Build graph representation of RCPSP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with tasks, precedence constraints, and resource constraints
    - Tasks (type 0): weighted by criticality (duration * resource demand)
    - Precedence constraints (type 1): weighted by impact on critical path
    - Resource constraints (type 1): weighted by tightness (utilization pressure)
    - Resource competition edges: tasks competing for same overloaded resources
    """
    
    # Extract basic data
    n_res = json_data.get('n_res', 0)
    n_tasks = json_data.get('n_tasks', 0)
    rc = json_data.get('rc', [])  # Resource capacities
    d = json_data.get('d', [])    # Task durations
    rr_flat = json_data.get('rr', [])  # Flattened resource requirements
    
    # Reconstruct 2D resource requirements array
    rr = []
    if len(rr_flat) == n_res * n_tasks:
        for r in range(n_res):
            rr.append(rr_flat[r * n_tasks:(r + 1) * n_tasks])
    else:
        # Fallback: assume uniform resource requirements
        rr = [[1 for _ in range(n_tasks)] for _ in range(n_res)]
    
    # Parse successor information from DZN file
    dzn_file = sys.argv[2] if len(sys.argv) > 2 else None
    successors = {}
    if dzn_file:
        successors = parse_dzn_successors(dzn_file)
    
    G = nx.Graph()
    
    # Calculate task criticality metrics
    max_duration = max(d) if d else 1
    max_resource_demand = 0
    task_resource_demands = []
    
    for t in range(n_tasks):
        total_demand = sum(rr[r][t] if r < len(rr) and t < len(rr[r]) else 0 for r in range(n_res))
        task_resource_demands.append(total_demand)
        max_resource_demand = max(max_resource_demand, total_demand)
    
    max_resource_demand = max(max_resource_demand, 1)
    
    # Add task nodes (type 0) with criticality-based weights
    for t in range(n_tasks):
        task_id = t + 1  # 1-indexed
        duration = d[t] if t < len(d) else 1
        resource_demand = task_resource_demands[t]
        
        # Weight combines duration impact and resource consumption
        duration_weight = duration / max_duration
        resource_weight = resource_demand / max_resource_demand
        
        # Non-linear combination emphasizing high-impact tasks
        criticality = math.sqrt(duration_weight * resource_weight)
        
        G.add_node(f'task_{task_id}', type=0, weight=criticality)
    
    # Add precedence constraint nodes (type 1)
    precedence_count = 0
    for task_id, succ_list in successors.items():
        for succ_id in succ_list:
            precedence_count += 1
            # Weight by duration impact of the dependency
            pred_duration = d[task_id - 1] if task_id <= len(d) else 1
            succ_duration = d[succ_id - 1] if succ_id <= len(d) else 1
            
            # Higher weight for dependencies involving longer tasks
            impact = (pred_duration + succ_duration) / (2 * max_duration)
            constraint_weight = min(impact * 1.5, 1.0)  # Scale up impact
            
            constraint_id = f'prec_{task_id}_{succ_id}'
            G.add_node(constraint_id, type=1, weight=constraint_weight)
            
            # Connect tasks to precedence constraint
            G.add_edge(f'task_{task_id}', constraint_id, weight=0.8)
            G.add_edge(f'task_{succ_id}', constraint_id, weight=0.8)
    
    # Add resource constraint nodes (type 1) with tightness-based weights
    total_capacity = sum(rc)
    
    for r in range(n_res):
        capacity = rc[r] if r < len(rc) else 1
        
        # Calculate total demand for this resource
        total_demand = sum(rr[r][t] if t < len(rr[r]) else 0 for t in range(n_tasks))
        
        # Resource tightness: how oversubscribed is this resource
        if total_demand > capacity:
            tightness = 1.0 - (capacity / total_demand)
        else:
            # Even if not oversubscribed, some tension exists
            tightness = 0.3 + 0.4 * (total_demand / capacity)
        
        tightness = min(tightness, 1.0)
        
        constraint_id = f'resource_{r + 1}'
        G.add_node(constraint_id, type=1, weight=tightness)
        
        # Connect tasks that use this resource
        for t in range(n_tasks):
            task_id = t + 1
            requirement = rr[r][t] if t < len(rr[r]) else 0
            
            if requirement > 0:
                # Edge weight based on how much of the resource this task consumes
                consumption_ratio = requirement / capacity if capacity > 0 else 0.5
                edge_weight = min(consumption_ratio * 2, 1.0)  # Scale up for visibility
                
                G.add_edge(f'task_{task_id}', constraint_id, weight=edge_weight)
    
    # Add resource competition edges between tasks
    # Tasks compete if they jointly would exceed resource capacity
    for r in range(n_res):
        capacity = rc[r] if r < len(rc) else 1
        
        # Find tasks that use this resource significantly
        heavy_users = []
        for t in range(n_tasks):
            requirement = rr[r][t] if t < len(rr[r]) else 0
            if requirement > capacity * 0.3:  # Tasks using >30% of resource
                heavy_users.append((t + 1, requirement))
        
        # Add competition edges between heavy users
        for i in range(len(heavy_users)):
            for j in range(i + 1, len(heavy_users)):
                task1_id, req1 = heavy_users[i]
                task2_id, req2 = heavy_users[j]
                
                # Only add edge if combined requirement exceeds capacity
                if req1 + req2 > capacity:
                    competition_strength = (req1 + req2) / (2 * capacity)
                    competition_strength = min(competition_strength, 1.0)
                    
                    G.add_edge(f'task_{task1_id}', f'task_{task2_id}', 
                             weight=competition_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()