#!/usr/bin/env python3
"""
Graph converter for Rectangle Packing Problem.
Created using subagent_prompt.md version: v_01

This problem involves packing N squares of sizes 1x1, 2x2, ..., NxN into the 
smallest rectangular container without overlaps. Key challenges include:
- Non-overlap constraints between squares of different sizes
- Boundary constraints for container dimensions
- Area utilization optimization
- Large squares dominate the packing difficulty
"""

import sys
import json
import networkx as nx
from pathlib import Path
import math


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the rectangle packing instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with explicit constraint nodes
    - Squares as type 0 nodes (weighted by area/dominance)
    - Constraints as type 1 nodes (weighted by tightness/scope)
    - Container dimensions as type 2 nodes (weighted by scarcity)
    
    Key insights:
    - Larger squares are exponentially more constraining
    - Packing density increases difficulty non-linearly
    - Corner/edge placement creates geometric conflicts
    - Unit square consideration affects problem complexity
    """
    # Access data from json_data
    n = json_data.get('n', 8)
    consider_unit = json_data.get('Consider_Unit_Square', True)
    
    G = nx.Graph()
    
    # Calculate problem metrics for weight calibration
    total_area = sum(i * i for i in range(1, n + 1))
    min_container_area = total_area  # Lower bound
    theoretical_side = math.sqrt(total_area)
    
    # Determine which squares to consider
    squares = []
    if consider_unit:
        squares = list(range(1, n + 1))
    else:
        squares = list(range(2, n + 1))
    
    # Type 0: Square nodes with size-based dominance weights
    max_square = n
    for i in squares:
        # Larger squares dominate exponentially (geometric packing difficulty)
        size_dominance = (i * i) / (max_square * max_square)
        # Add geometric placement difficulty (larger squares have fewer valid positions)
        placement_difficulty = math.exp(2.0 * i / max_square) / math.e**2
        weight = min(0.5 * size_dominance + 0.5 * placement_difficulty, 1.0)
        G.add_node(f'square_{i}', type=0, weight=weight)
    
    # Type 2: Container dimension nodes (width and height as resources)
    # These represent the scarce spatial resources
    min_width = math.ceil(theoretical_side)
    min_height = n if n % 2 == 1 else n + 1  # From MZN model
    
    # Width resource scarcity
    width_utilization = total_area / (min_width * min_height)
    width_scarcity = min(width_utilization * 1.2, 1.0)
    G.add_node('width_resource', type=2, weight=width_scarcity)
    
    # Height resource scarcity  
    height_scarcity = min(width_utilization * 1.1, 1.0)
    G.add_node('height_resource', type=2, weight=height_scarcity)
    
    # Type 1: Constraint nodes with scope and tightness-based weights
    
    # 1. Non-overlap constraints (one per square pair)
    constraint_id = 0
    for i in range(len(squares)):
        for j in range(i + 1, len(squares)):
            sq1, sq2 = squares[i], squares[j]
            # Tightness based on size similarity and placement conflicts
            size_conflict = min(sq1, sq2) / max(sq1, sq2)  # Similar sizes = higher conflict
            area_pressure = ((sq1 * sq1) + (sq2 * sq2)) / total_area
            tightness = min(0.4 * size_conflict + 0.6 * area_pressure, 1.0)
            
            constraint_node = f'nonoverlap_{sq1}_{sq2}'
            G.add_node(constraint_node, type=1, weight=tightness)
            G.add_edge(f'square_{sq1}', constraint_node, weight=0.8)
            G.add_edge(f'square_{sq2}', constraint_node, weight=0.8)
            constraint_id += 1
    
    # 2. Boundary constraints (squares must fit within container)
    for i in squares:
        # X-boundary constraint
        boundary_pressure = i / theoretical_side  # Larger squares more constrained
        x_boundary_weight = min(boundary_pressure * 1.5, 1.0)
        x_constraint = f'x_boundary_{i}'
        G.add_node(x_constraint, type=1, weight=x_boundary_weight)
        G.add_edge(f'square_{i}', x_constraint, weight=0.9)
        G.add_edge('width_resource', x_constraint, weight=0.7)
        
        # Y-boundary constraint
        y_boundary_weight = min(boundary_pressure * 1.4, 1.0)
        y_constraint = f'y_boundary_{i}'
        G.add_node(y_constraint, type=1, weight=y_boundary_weight)
        G.add_edge(f'square_{i}', y_constraint, weight=0.9)
        G.add_edge('height_resource', y_constraint, weight=0.7)
    
    # 3. Area utilization constraint (global)
    area_tightness = min(total_area / min_container_area, 1.0)
    G.add_node('area_constraint', type=1, weight=area_tightness)
    # Connect all squares to area constraint
    for i in squares:
        area_contribution = (i * i) / total_area
        G.add_edge(f'square_{i}', 'area_constraint', weight=area_contribution)
    G.add_edge('width_resource', 'area_constraint', weight=0.6)
    G.add_edge('height_resource', 'area_constraint', weight=0.6)
    
    # 4. Cumulative constraints (from MZN model - resource scheduling-like)
    # These ensure "non-overload" in each dimension
    cumul_tightness = min(1.2 * width_utilization - 0.2, 1.0)
    
    # X-cumulative constraint
    G.add_node('x_cumulative', type=1, weight=cumul_tightness)
    for i in squares:
        resource_demand = i / sum(squares)  # Relative demand
        G.add_edge(f'square_{i}', 'x_cumulative', weight=resource_demand)
    G.add_edge('height_resource', 'x_cumulative', weight=0.8)
    
    # Y-cumulative constraint
    G.add_node('y_cumulative', type=1, weight=cumul_tightness)
    for i in squares:
        resource_demand = i / sum(squares)
        G.add_edge(f'square_{i}', 'y_cumulative', weight=resource_demand)
    G.add_edge('width_resource', 'y_cumulative', weight=0.8)
    
    # 5. Symmetry-breaking constraints (affects largest square)
    if n in squares:
        # The largest square has symmetry-breaking constraints
        sym_weight = 0.3  # These are less tight but important for search
        
        G.add_node('symmetry_x', type=1, weight=sym_weight)
        G.add_edge(f'square_{n}', 'symmetry_x', weight=0.5)
        G.add_edge('width_resource', 'symmetry_x', weight=0.4)
        
        G.add_node('symmetry_y', type=1, weight=sym_weight)
        G.add_edge(f'square_{n}', 'symmetry_y', weight=0.5)
        G.add_edge('height_resource', 'symmetry_y', weight=0.4)
    
    # Add high-conflict edges for competing large squares
    # Large squares compete heavily for prime positions (corners, edges)
    large_squares = [i for i in squares if i >= n * 0.6]
    for i in range(len(large_squares)):
        for j in range(i + 1, len(large_squares)):
            sq1, sq2 = large_squares[i], large_squares[j]
            # Direct conflict for spatial resources
            conflict_strength = ((sq1 * sq1) + (sq2 * sq2)) / (2 * total_area)
            if conflict_strength > 0.3:  # Only add if significant
                G.add_edge(f'square_{sq1}', f'square_{sq2}', 
                          weight=min(conflict_strength * 2, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()