#!/usr/bin/env python3
"""
Graph converter for Rotating Workforce Scheduling problem.
Created using subagent_prompt.md version: v_02

This problem is about assigning employees to rotating shift schedules while meeting daily staffing requirements.
Key challenges: complex constraints (consecutive days off, weekend requirements, night shift limits), balancing workload,
meeting varying daily/shift requirements.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the rotating workforce scheduling problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling employees, shifts, and constraints
    - Employee nodes (type 0): represent employees making scheduling decisions
    - Shift requirement nodes (type 1): represent daily shift requirements that must be met
    - Complex constraint nodes (type 1): represent weekend, consecutive days off, and night shift restrictions
    - Edges connect employees to shifts they could work and constraints they participate in
    
    What makes instances hard:
    - High staffing requirements relative to employee count
    - Uneven distribution of requirements across days/shifts
    - Night shift concentration (harder to satisfy constraints)
    - Weekend requirements (limited by weekend-off constraints)
    """
    # Access data directly from json_data dict
    employees = json_data.get('employees', 0)
    requirements_flat = json_data.get('requirements', [])
    
    # Reshape requirements array from flat to 7 days × 3 shifts
    days = 7
    shifts = 3
    if len(requirements_flat) != days * shifts:
        # Fallback for malformed data
        requirements = [[1, 1, 1] for _ in range(days)]
    else:
        requirements = []
        for day in range(days):
            day_reqs = requirements_flat[day*shifts:(day+1)*shifts]
            requirements.append(day_reqs)
    
    G = nx.Graph()
    
    # Calculate some metrics for weight calculations
    total_requirements = sum(requirements_flat) if requirements_flat else employees
    max_daily_req = max(sum(requirements[day]) for day in range(days)) if requirements else 1
    avg_req = total_requirements / (days * shifts) if days * shifts > 0 else 1
    
    # Employee nodes (type 0) - decision makers
    # Weight by workload pressure: higher when total requirements are high relative to workforce
    workload_pressure = min(total_requirements / (employees * days) if employees > 0 else 1.0, 1.0)
    for emp in range(employees):
        # Slight variation in employee weights to model different skill levels/availability
        emp_weight = workload_pressure * (0.8 + 0.4 * (emp % 3) / 2)  # 0.8-1.2 range, then clamped
        G.add_node(f'employee_{emp}', type=0, weight=min(emp_weight, 1.0))
    
    # Shift requirement constraint nodes (type 1) - one per day-shift combination
    # Weight by requirement tightness and criticality
    day_names = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun']
    shift_names = ['Day', 'Evening', 'Night']
    
    for day in range(days):
        for shift in range(shifts):
            req = requirements[day][shift] if day < len(requirements) and shift < len(requirements[day]) else 1
            
            # Tightness: how demanding this requirement is
            tightness = min(req / employees if employees > 0 else 0.5, 1.0)
            
            # Night shifts and weekend shifts are typically harder to fill
            criticality = 1.0
            if shift == 2:  # Night shift
                criticality *= 1.3
            if day >= 5:  # Weekend (Sat, Sun)
                criticality *= 1.2
                
            # Combined weight with exponential scaling for high requirements
            weight = min(tightness * criticality * (1 + math.log(1 + req / avg_req)), 1.0)
            
            constraint_id = f'shift_req_{day_names[day]}_{shift_names[shift]}'
            G.add_node(constraint_id, type=1, weight=weight)
    
    # Complex constraint nodes (type 1) - representing the hard constraints
    
    # Weekend-off constraint (1 out of 3 weekends must be free)
    weekend_pressure = sum(requirements[5]) + sum(requirements[6]) if len(requirements) >= 7 else 2
    weekend_weight = min(weekend_pressure / (employees * 0.33) if employees > 0 else 0.7, 1.0)
    G.add_node('weekend_off_constraint', type=1, weight=weekend_weight)
    
    # Consecutive days off constraint (at least 2 consecutive days off per week)
    consecutive_weight = 0.8  # Moderately important
    G.add_node('consecutive_off_constraint', type=1, weight=consecutive_weight)
    
    # Night shift limitation constraint (at most 2 consecutive night shifts)
    night_total = sum(requirements[day][2] for day in range(days))
    night_weight = min(night_total / (employees * 0.4) if employees > 0 else 0.6, 1.0)
    G.add_node('night_limit_constraint', type=1, weight=night_weight)
    
    # Rest requirement constraint (at most 5 days without rest)
    rest_weight = 0.9  # High importance for worker welfare
    G.add_node('rest_requirement_constraint', type=1, weight=rest_weight)
    
    # Add edges: employees to shift requirements
    # Edge weight represents how much this employee contributes to meeting this requirement
    for emp in range(employees):
        for day in range(days):
            for shift in range(shifts):
                req = requirements[day][shift] if day < len(requirements) and shift < len(requirements[day]) else 1
                
                # Contribution weight: higher when requirement is larger (more critical to fill)
                # Use logarithmic scaling to prevent extreme values
                contribution = math.log(1 + req) / math.log(1 + max_daily_req) if max_daily_req > 0 else 0.5
                
                # Night shifts have higher edge weights (harder to assign)
                if shift == 2:  # Night shift
                    contribution *= 1.2
                
                constraint_id = f'shift_req_{day_names[day]}_{shift_names[shift]}'
                G.add_edge(f'employee_{emp}', constraint_id, weight=min(contribution, 1.0))
    
    # Add edges: employees to complex constraints
    # All employees participate in these constraints equally
    base_constraint_edge_weight = 0.7
    
    for emp in range(employees):
        G.add_edge(f'employee_{emp}', 'weekend_off_constraint', weight=base_constraint_edge_weight)
        G.add_edge(f'employee_{emp}', 'consecutive_off_constraint', weight=base_constraint_edge_weight)
        G.add_edge(f'employee_{emp}', 'night_limit_constraint', weight=base_constraint_edge_weight)
        G.add_edge(f'employee_{emp}', 'rest_requirement_constraint', weight=base_constraint_edge_weight)
    
    # Add conflict edges between high-demand shifts that compete for the same employees
    # This models the difficulty of satisfying multiple high-demand shifts
    high_demand_threshold = avg_req * 1.5
    high_demand_shifts = []
    
    for day in range(days):
        for shift in range(shifts):
            req = requirements[day][shift] if day < len(requirements) and shift < len(requirements[day]) else 1
            if req > high_demand_threshold:
                high_demand_shifts.append((day, shift, req))
    
    # Add conflict edges between competing high-demand shifts
    for i, (day1, shift1, req1) in enumerate(high_demand_shifts):
        for j, (day2, shift2, req2) in enumerate(high_demand_shifts[i+1:], i+1):
            # Conflicts are stronger for:
            # 1. Adjacent days (harder to satisfy both)
            # 2. Same shift type (compete for same skill set)
            # 3. Higher total demand
            
            conflict_strength = 0.3  # Base conflict
            
            if abs(day1 - day2) <= 1:  # Adjacent or same day
                conflict_strength += 0.3
            if shift1 == shift2:  # Same shift type
                conflict_strength += 0.2
            
            # Scale by total demand intensity
            demand_factor = (req1 + req2) / (2 * avg_req) if avg_req > 0 else 1
            conflict_strength *= min(demand_factor, 2.0)
            
            constraint1 = f'shift_req_{day_names[day1]}_{shift_names[shift1]}'
            constraint2 = f'shift_req_{day_names[day2]}_{shift_names[shift2]}'
            G.add_edge(constraint1, constraint2, weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()