#!/usr/bin/env python3
"""
Graph converter for rotating-workforce problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling workers across different shifts with rotation constraints.
Key challenges: satisfying temporal requirements, enforcing work/off block constraints, avoiding forbidden sequences.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the rotating workforce scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the scheduling structure
    - Worker-day combinations as decision variables (type 0)
    - Shift requirements, block constraints, and forbidden sequences as constraints (type 1)
    - Shifts as resources (type 2)
    - Connect based on participation, conflicts, and resource usage
    """
    # Extract problem parameters
    week_length = json_data.get('week_length', 7)
    nb_workers = json_data.get('nb_workers', 0)
    nb_shifts = json_data.get('nb_shifts', 0)
    min_daysoff = json_data.get('min_daysoff', 1)
    max_daysoff = json_data.get('max_daysoff', 7)
    min_work = json_data.get('min_work', 1)
    max_work = json_data.get('max_work', 7)
    nb_forbidden = json_data.get('nb_forbidden', 0)
    
    # Shift data
    shift_block_min = json_data.get('shift_block_min', [])
    shift_block_max = json_data.get('shift_block_max', [])
    temp_req = json_data.get('temp_req', [])
    forbidden_before = json_data.get('forbidden_before', [])
    forbidden_after = json_data.get('forbidden_after', [])
    
    G = nx.Graph()
    
    # Calculate derived parameters for weighting
    total_worker_days = week_length * nb_workers
    total_temp_req = sum(temp_req) if temp_req else 0
    workload_ratio = total_temp_req / total_worker_days if total_worker_days > 0 else 0.5
    
    # Type 0: Worker-day decision variables
    # Weight by: centrality in week, workload pressure
    for worker in range(nb_workers):
        for day in range(week_length):
            # Central days and workers have higher complexity
            day_centrality = 1.0 - abs(day - week_length//2) / (week_length//2 + 1)
            worker_centrality = 1.0 - abs(worker - nb_workers//2) / (nb_workers//2 + 1)
            
            # Days with high requirements are more constrained
            day_req_sum = 0
            if temp_req:
                for shift in range(nb_shifts):
                    req_idx = shift * week_length + day
                    if req_idx < len(temp_req):
                        day_req_sum += temp_req[req_idx]
            
            day_pressure = day_req_sum / nb_workers if nb_workers > 0 else 0.5
            
            # Combine factors with non-linear weighting
            weight = 0.3 * day_centrality + 0.3 * worker_centrality + 0.4 * math.sqrt(day_pressure)
            weight = min(max(weight, 0.1), 1.0)
            
            G.add_node(f'worker_{worker}_day_{day}', type=0, weight=weight)
    
    # Type 2: Shift resources
    # Weight by: scarcity, flexibility constraints
    max_block_size = max(shift_block_max) if shift_block_max else 1
    for shift in range(nb_shifts):
        # Calculate shift requirements
        shift_total_req = 0
        if temp_req:
            for day in range(week_length):
                req_idx = shift * week_length + day
                if req_idx < len(temp_req):
                    shift_total_req += temp_req[req_idx]
        
        # Scarcity based on total requirement vs availability
        scarcity = shift_total_req / total_worker_days if total_worker_days > 0 else 0.5
        
        # Flexibility penalty for strict block constraints
        block_min = shift_block_min[shift] if shift < len(shift_block_min) else 1
        block_max = shift_block_max[shift] if shift < len(shift_block_max) else 7
        flexibility = 1.0 - (block_min / block_max) if block_max > 0 else 0.5
        
        # Non-linear combination emphasizing scarcity
        weight = 0.7 * math.exp(scarcity) / math.e + 0.3 * (1.0 - flexibility)
        weight = min(max(weight, 0.1), 1.0)
        
        G.add_node(f'shift_{shift}', type=2, weight=weight)
    
    # Type 1: Temporal requirement constraints (one per shift-day combination)
    for shift in range(nb_shifts):
        for day in range(week_length):
            req_idx = shift * week_length + day
            requirement = temp_req[req_idx] if req_idx < len(temp_req) else 0
            
            # Tightness based on requirement vs available workers
            tightness = requirement / nb_workers if nb_workers > 0 else 0.5
            # Higher requirements create tighter constraints
            weight = min(math.sqrt(tightness) + 0.2, 1.0)
            
            constraint_id = f'temp_req_shift_{shift}_day_{day}'
            G.add_node(constraint_id, type=1, weight=weight)
            
            # Connect to relevant worker-day variables
            for worker in range(nb_workers):
                var_id = f'worker_{worker}_day_{day}'
                # Edge weight based on requirement intensity
                edge_weight = min(tightness + 0.3, 1.0)
                G.add_edge(var_id, constraint_id, weight=edge_weight)
            
            # Connect to shift resource
            resource_edge_weight = min(tightness + 0.2, 1.0)
            G.add_edge(f'shift_{shift}', constraint_id, weight=resource_edge_weight)
    
    # Type 1: Work block constraints
    # Minimum work block constraint
    work_block_tightness = min_work / max_work if max_work > 0 else 0.5
    min_work_weight = 0.6 + 0.4 * work_block_tightness
    G.add_node('min_work_block_constraint', type=1, weight=min_work_weight)
    
    # Maximum work block constraint  
    max_work_weight = 0.4 + 0.6 * (1.0 - work_block_tightness)
    G.add_node('max_work_block_constraint', type=1, weight=max_work_weight)
    
    # Type 1: Days-off block constraints
    daysoff_tightness = min_daysoff / max_daysoff if max_daysoff > 0 else 0.5
    min_daysoff_weight = 0.5 + 0.5 * daysoff_tightness
    G.add_node('min_daysoff_constraint', type=1, weight=min_daysoff_weight)
    
    max_daysoff_weight = 0.3 + 0.7 * (1.0 - daysoff_tightness)
    G.add_node('max_daysoff_constraint', type=1, weight=max_daysoff_weight)
    
    # Connect work/daysoff constraints to all worker-day variables
    # (they affect global scheduling patterns)
    for worker in range(nb_workers):
        for day in range(week_length):
            var_id = f'worker_{worker}_day_{day}'
            # Stronger connections for variables at sequence boundaries
            boundary_weight = 0.3
            if day == 0 or day == week_length - 1:
                boundary_weight = 0.7
            
            G.add_edge(var_id, 'min_work_block_constraint', weight=boundary_weight)
            G.add_edge(var_id, 'max_work_block_constraint', weight=boundary_weight)
            G.add_edge(var_id, 'min_daysoff_constraint', weight=boundary_weight)
            G.add_edge(var_id, 'max_daysoff_constraint', weight=boundary_weight)
    
    # Type 1: Forbidden sequence constraints
    for i in range(nb_forbidden):
        if i < len(forbidden_before) and i < len(forbidden_after):
            before_shift = forbidden_before[i] - 1  # Convert to 0-indexed
            after_shift = forbidden_after[i] - 1    # Convert to 0-indexed
            
            if 0 <= before_shift < nb_shifts and 0 <= after_shift < nb_shifts:
                # Weight based on how restrictive this forbidden sequence is
                # More restrictions for shifts with higher requirements
                before_req = sum(temp_req[before_shift * week_length + d] 
                               for d in range(week_length) 
                               if before_shift * week_length + d < len(temp_req))
                after_req = sum(temp_req[after_shift * week_length + d] 
                              for d in range(week_length) 
                              if after_shift * week_length + d < len(temp_req))
                
                restriction_impact = (before_req + after_req) / (2 * total_worker_days) if total_worker_days > 0 else 0.5
                weight = 0.4 + 0.6 * restriction_impact
                
                constraint_id = f'forbidden_seq_{before_shift}_to_{after_shift}'
                G.add_node(constraint_id, type=1, weight=weight)
                
                # Connect to relevant shifts
                G.add_edge(f'shift_{before_shift}', constraint_id, weight=0.8)
                G.add_edge(f'shift_{after_shift}', constraint_id, weight=0.8)
                
                # Connect to worker-day variables that could be affected
                # (consecutive days across the schedule)
                for worker in range(nb_workers):
                    for day in range(week_length - 1):
                        var1_id = f'worker_{worker}_day_{day}'
                        var2_id = f'worker_{worker}_day_{day + 1}'
                        # Weaker connection since this is a conditional constraint
                        G.add_edge(var1_id, constraint_id, weight=0.3)
                        G.add_edge(var2_id, constraint_id, weight=0.3)
    
    # Add shift-specific block constraints
    for shift in range(nb_shifts):
        if shift < len(shift_block_min) and shift < len(shift_block_max):
            block_min = shift_block_min[shift]
            block_max = shift_block_max[shift]
            
            # Shift-specific minimum block constraint
            min_constraint_id = f'shift_{shift}_min_block'
            min_weight = 0.5 + 0.5 * (block_min / max_block_size)
            G.add_node(min_constraint_id, type=1, weight=min_weight)
            G.add_edge(f'shift_{shift}', min_constraint_id, weight=0.9)
            
            # Shift-specific maximum block constraint
            max_constraint_id = f'shift_{shift}_max_block'
            max_weight = 0.4 + 0.6 * (1.0 - block_max / max_block_size)
            G.add_node(max_constraint_id, type=1, weight=max_weight)
            G.add_edge(f'shift_{shift}', max_constraint_id, weight=0.9)
            
            # Connect to worker-day variables (affecting consecutive scheduling)
            for worker in range(nb_workers):
                for day in range(week_length):
                    var_id = f'worker_{worker}_day_{day}'
                    # Edge weight based on how restrictive the block constraints are
                    restrictiveness = (block_min + max_block_size - block_max) / (2 * max_block_size)
                    edge_weight = 0.2 + 0.5 * restrictiveness
                    
                    G.add_edge(var_id, min_constraint_id, weight=edge_weight)
                    G.add_edge(var_id, max_constraint_id, weight=edge_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()