#!/usr/bin/env python3
"""
Graph converter for Solitaire Battleships problem.
Created using subagent_prompt.md version: v_02

This problem is about placing ships on a grid with row/column constraints.
Key challenges: ship shape constraints, separation requirements, and satisfying row/column sums.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Solitaire Battleships problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Bipartite graph with cells as variables and constraints as explicit nodes
    - Grid cells are decision variables (type 0) with position-based weights
    - Row/column sum constraints are explicit constraint nodes (type 1)
    - Ship shape and separation constraints are modeled as constraint nodes (type 1)
    - Hint constraints for fixed pieces create additional constraint nodes
    - Edge weights reflect constraint tightness and cell importance
    """
    # Access data from json_data dict
    width = json_data.get('width', 10)
    height = json_data.get('height', 10)
    maxship = json_data.get('maxship', 4)
    ship = json_data.get('ship', [4, 3, 2, 1])  # Default battleship configuration
    rowsum = json_data.get('rowsum', [0] * width)
    colsum = json_data.get('colsum', [0] * height)
    hint = json_data.get('hint', [0] * (width * height))
    
    # Create graph
    G = nx.Graph()
    
    # Variable nodes: grid cells (type 0)
    # Weight based on centrality and constraint potential
    for i in range(width):
        for j in range(height):
            cell_id = f'cell_{i}_{j}'
            
            # Central cells are more constrained due to ship placement options
            centrality = 1.0 - (abs(i - width//2) + abs(j - height//2)) / (width + height)
            
            # Cells with hints are more constrained
            hint_idx = i * height + j
            has_hint = hint_idx < len(hint) and hint[hint_idx] != 0
            hint_factor = 1.5 if has_hint else 1.0
            
            # Normalize weight to [0,1]
            weight = min(centrality * hint_factor, 1.0)
            
            G.add_node(cell_id, type=0, weight=weight)
    
    # Constraint nodes: Row sum constraints (type 1)
    max_row_density = max(rowsum) / height if height > 0 else 0
    for i in range(width):
        constraint_id = f'row_sum_{i}'
        # Tightness based on how full the row needs to be
        density = rowsum[i] / height if height > 0 else 0
        tightness = density / max_row_density if max_row_density > 0 else 0.5
        
        G.add_node(constraint_id, type=1, weight=tightness)
        
        # Connect to all cells in this row
        for j in range(height):
            cell_id = f'cell_{i}_{j}'
            # Edge weight based on how much this cell contributes to row constraint
            edge_weight = min(density * 1.5, 1.0)
            G.add_edge(cell_id, constraint_id, weight=edge_weight)
    
    # Constraint nodes: Column sum constraints (type 1)
    max_col_density = max(colsum) / width if width > 0 else 0
    for j in range(height):
        constraint_id = f'col_sum_{j}'
        # Tightness based on how full the column needs to be
        density = colsum[j] / width if width > 0 else 0
        tightness = density / max_col_density if max_col_density > 0 else 0.5
        
        G.add_node(constraint_id, type=1, weight=tightness)
        
        # Connect to all cells in this column
        for i in range(width):
            cell_id = f'cell_{i}_{j}'
            # Edge weight based on how much this cell contributes to column constraint
            edge_weight = min(density * 1.5, 1.0)
            G.add_edge(cell_id, constraint_id, weight=edge_weight)
    
    # Constraint nodes: Ship type constraints (type 1)
    total_ship_pieces = sum(ship[s-1] * s for s in range(1, min(len(ship)+1, maxship+1)))
    total_cells = width * height
    overall_density = total_ship_pieces / total_cells if total_cells > 0 else 0
    
    for s in range(1, min(len(ship)+1, maxship+1)):
        if s-1 < len(ship) and ship[s-1] > 0:
            constraint_id = f'ship_type_{s}'
            # Longer ships are harder to place
            difficulty = math.exp(s / maxship) / math.e if maxship > 0 else 0.5
            # Number of ships affects difficulty
            count_factor = min(ship[s-1] / 10.0, 1.0)  # Normalize assuming max 10 ships
            weight = min(difficulty * count_factor * overall_density, 1.0)
            
            G.add_node(constraint_id, type=1, weight=weight)
            
            # Connect to cells that could be part of ships of this length
            # For simplicity, connect to all cells with weight based on placement potential
            for i in range(width):
                for j in range(height):
                    cell_id = f'cell_{i}_{j}'
                    
                    # Can this cell be part of a ship of length s?
                    can_place_horizontal = j + s <= height
                    can_place_vertical = i + s <= width
                    
                    if can_place_horizontal or can_place_vertical:
                        # Weight based on how many ways this cell can participate
                        placement_options = int(can_place_horizontal) + int(can_place_vertical)
                        edge_weight = min(placement_options / 2.0 * difficulty, 1.0)
                        G.add_edge(cell_id, constraint_id, weight=edge_weight)
    
    # Constraint nodes: Separation constraints (type 1)
    # Model the global separation constraint that ships cannot touch
    separation_constraint_id = 'separation_constraint'
    # Weight based on ship density - more ships means tighter separation constraint
    separation_weight = min(overall_density * 2.0, 1.0)
    G.add_node(separation_constraint_id, type=1, weight=separation_weight)
    
    # Connect cells to separation constraint with weights based on neighborhood density
    for i in range(width):
        for j in range(height):
            cell_id = f'cell_{i}_{j}'
            
            # Count potential neighboring cells (8-connected neighborhood)
            neighbors = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 0 <= ni < width and 0 <= nj < height:
                        neighbors += 1
            
            # Cells with more neighbors are more constrained by separation
            constraint_strength = neighbors / 8.0  # Max 8 neighbors
            edge_weight = min(constraint_strength * separation_weight, 1.0)
            G.add_edge(cell_id, separation_constraint_id, weight=edge_weight)
    
    # Constraint nodes: Hint constraints (type 1)
    # Create individual constraint nodes for each hint
    for i in range(width):
        for j in range(height):
            hint_idx = i * height + j
            if hint_idx < len(hint) and hint[hint_idx] != 0:
                cell_id = f'cell_{i}_{j}'
                constraint_id = f'hint_{i}_{j}'
                
                # Hints are hard constraints - high weight
                G.add_node(constraint_id, type=1, weight=1.0)
                # Strong connection to the constrained cell
                G.add_edge(cell_id, constraint_id, weight=1.0)
    
    # Add conflict edges between cells that cannot both contain ships due to separation
    # This captures direct incompatibilities beyond the separation constraint node
    for i in range(width):
        for j in range(height):
            cell_id = f'cell_{i}_{j}'
            
            # Add conflict edges to adjacent cells (ships cannot touch)
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 0 <= ni < width and 0 <= nj < height:
                        neighbor_id = f'cell_{ni}_{nj}'
                        if neighbor_id > cell_id:  # Avoid duplicate edges
                            # Diagonal conflicts are weaker than orthogonal
                            is_diagonal = abs(di) + abs(dj) == 2
                            conflict_weight = 0.3 if is_diagonal else 0.6
                            # Scale by overall density - denser puzzles have tighter conflicts
                            conflict_weight *= min(overall_density * 2.0, 1.0)
                            G.add_edge(cell_id, neighbor_id, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()