#!/usr/bin/env python3
"""
Graph converter for search_stress problem.
Created using subagent_prompt.md version: v_02

This problem is about a deliberately unsatisfiable graph coloring stress test.
Key challenges: Complex constraint interactions designed to cause search failure.
The problem creates n sequential subgraphs with forced color equalities that 
ultimately conflict, making the instance unsatisfiable.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the search_stress problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the constraint structure that creates search stress
    - Variables are colored nodes in sequential subgraphs
    - Constraints enforce color conflicts and equalities
    - The bipartite structure captures propagation bottlenecks
    - Weights reflect constraint tightness and variable criticality
    """
    n = json_data.get('n', 4)  # number of subgraph copies
    k = json_data.get('k', 4)  # subgraph size and color count
    
    G = nx.Graph()
    
    # Calculate total variables: n*k+1
    total_vars = n * k + 1
    
    # Add variable nodes (type 0) - representing colored vertices
    for i in range(1, total_vars + 1):
        # Calculate position-based criticality
        if i == 1 or i == total_vars:  # First and last nodes are most critical
            criticality = 1.0
        elif i % (k + 1) == 1:  # Connecting nodes between subgraphs
            criticality = 0.8
        else:  # Internal subgraph nodes
            # More central positions in subgraph are more constrained
            subgraph_pos = ((i - 2) % k) + 1
            centrality = 1.0 - abs(subgraph_pos - k/2) / (k/2)
            criticality = 0.4 + 0.4 * centrality
            
        G.add_node(f'x_{i}', type=0, weight=criticality)
    
    # Add constraint nodes (type 1) for each constraint type
    constraint_id = 0
    
    # Constraints within each subgraph (i = 1..n)
    for i in range(1, n + 1):
        base_idx = (i - 1) * k
        
        # Inequality constraints: x[(i-1)*k+1] != x[(i-1)*k+j] for j in 2..k
        for j in range(2, k + 1):
            constraint_id += 1
            # These constraints become tighter as k increases (more conflicts)
            tightness = min(1.0, (k - 1) / k)  # Approaches 1 as k grows
            G.add_node(f'neq_{i}_{j}', type=1, weight=tightness)
            
            # Connect variables to constraint
            G.add_edge(f'x_{base_idx + 1}', f'neq_{i}_{j}', weight=1.0)
            G.add_edge(f'x_{base_idx + j}', f'neq_{i}_{j}', weight=1.0)
        
        # Inequality constraints: x[i*k+1] != x[(i-1)*k + j] for j in 2..k
        for j in range(2, k + 1):
            constraint_id += 1
            tightness = min(1.0, (k - 1) / k)
            G.add_node(f'neq2_{i}_{j}', type=1, weight=tightness)
            
            # Connect variables to constraint
            G.add_edge(f'x_{i * k + 1}', f'neq2_{i}_{j}', weight=1.0)
            G.add_edge(f'x_{base_idx + j}', f'neq2_{i}_{j}', weight=1.0)
        
        # All-different constraint for x[(i-1)*k + j] where j in 2..k
        if k > 2:  # Only if there are enough variables
            constraint_id += 1
            # All-different becomes exponentially harder with more variables
            scope_size = k - 1
            tightness = 1.0 - math.exp(-scope_size / k)  # Non-linear scaling
            G.add_node(f'alldiff_{i}', type=1, weight=tightness)
            
            # Connect all variables in the scope
            edge_weight = 1.0 / scope_size  # Distribute weight among participants
            for j in range(2, k + 1):
                G.add_edge(f'x_{base_idx + j}', f'alldiff_{i}', weight=edge_weight)
    
    # Global conflict constraint: x[1] != x[n*k+1] 
    # This is the constraint that makes the problem unsatisfiable
    constraint_id += 1
    G.add_node('global_conflict', type=1, weight=1.0)  # Maximum tightness
    G.add_edge('x_1', 'global_conflict', weight=1.0)
    G.add_edge(f'x_{total_vars}', 'global_conflict', weight=1.0)
    
    # Add resource nodes (type 2) representing color domains
    for color in range(1, k + 1):
        # Color scarcity increases with problem size and constraint density
        demand_ratio = total_vars / k  # How many variables per color
        scarcity = min(1.0, demand_ratio / k)  # Normalized scarcity
        G.add_node(f'color_{color}', type=2, weight=scarcity)
        
        # Connect variables to their potential colors with exponential decay
        for var_idx in range(1, total_vars + 1):
            # Variables closer to their "natural" color have stronger connections
            natural_color = ((var_idx - 1) % k) + 1
            color_distance = min(abs(color - natural_color), k - abs(color - natural_color))
            connection_strength = math.exp(-2.0 * color_distance / k)
            
            if connection_strength > 0.1:  # Only add meaningful connections
                G.add_edge(f'x_{var_idx}', f'color_{color}', weight=connection_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()