#!/usr/bin/env python3
"""
Graph converter for search_stress2 problem.
Created using subagent_prompt.md version: v_02

This problem is a search stress test that creates chains of complex equality constraints.
Key challenges: Sequential constraint dependencies, different predicate complexities, and search space explosion.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the search_stress2 problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the sequential constraint chain structure
    - Variable nodes: t[i,j] variables with position-based weights
    - Constraint nodes: Each equality predicate instance (eq1-eq4)
    - Edge weights: Based on constraint complexity and variable participation
    - Chain structure captures the sequential dependencies between constraints
    """
    m = json_data.get('m', 2)  # number of copies of equality
    n = json_data.get('n', 7)  # size of domains
    
    G = nx.Graph()
    
    # Variable nodes: t[i,j] for i in 1..m, j in 0..n
    # Weight variables based on their role and position in the constraint chain
    for i in range(1, m + 1):
        for j in range(0, n + 1):
            var_id = f't_{i}_{j}'
            
            # Weight based on position in chain and domain position
            # t[i,0] variables are more critical as they connect constraints
            if j == 0:
                # Critical linking variables - higher weight for middle of chain
                chain_centrality = 1.0 - abs(i - m/2) / (m/2) if m > 1 else 1.0
                weight = 0.7 + 0.3 * chain_centrality
            else:
                # Auxiliary variables - weight decreases with j position
                aux_weight = 1.0 - (j - 1) / n if n > 1 else 0.5
                weight = 0.3 + 0.4 * aux_weight
            
            G.add_node(var_id, type=0, weight=weight)
    
    # Constraint nodes: One for each equality predicate instance
    # Each constraint connects consecutive rows based on predicate type
    predicate_complexity = {1: 0.9, 2: 0.7, 3: 0.8, 4: 0.6}  # Relative complexity
    
    for i in range(1, m):
        # Determine predicate type based on position (cycling through eq1-eq4)
        pred_type = ((i - 1) % 4) + 1
        constraint_id = f'eq{pred_type}_{i}'
        
        # Weight based on predicate complexity and position in chain
        base_complexity = predicate_complexity[pred_type]
        
        # Middle constraints are more critical (bottlenecks)
        position_factor = 1.0 - abs(i - m/2) / (m/2) if m > 1 else 1.0
        
        # Scale complexity with domain size (larger n = harder)
        size_factor = min(1.0, n / 10.0)
        
        weight = base_complexity * (0.6 + 0.4 * position_factor) * (0.7 + 0.3 * size_factor)
        G.add_node(constraint_id, type=1, weight=weight)
    
    # Additional constraint for t[1,0] != t[m,0]
    if m > 1:
        inequality_weight = 0.8  # High importance as it prevents trivial solutions
        G.add_node('neq_boundary', type=1, weight=inequality_weight)
    
    # Edges: Variable-constraint participation
    for i in range(1, m):
        constraint_id = f'eq{((i - 1) % 4) + 1}_{i}'
        
        # Each constraint involves t[i,0], t[i+1,0], and t[i,1]..t[i,n]
        # Connect primary variables (t[i,0] and t[i+1,0])
        G.add_edge(f't_{i}_0', constraint_id, weight=1.0)  # High participation
        G.add_edge(f't_{i+1}_0', constraint_id, weight=1.0)
        
        # Connect auxiliary variables t[i,j] for j in 1..n
        for j in range(1, n + 1):
            participation_weight = 0.8 - 0.1 * (j - 1) / n if n > 1 else 0.8
            G.add_edge(f't_{i}_{j}', constraint_id, weight=participation_weight)
    
    # Boundary inequality constraint edges
    if m > 1:
        G.add_edge('t_1_0', 'neq_boundary', weight=0.9)
        G.add_edge(f't_{m}_0', 'neq_boundary', weight=0.9)
    
    # Add resource nodes representing the domain constraints
    # Each domain position can be seen as a limited resource
    for val in range(0, n + 1):
        resource_id = f'domain_{val}'
        # Domain 0 is special (used in many constraints), others are regular values
        if val == 0:
            weight = 0.9  # High contention for value 0
        else:
            weight = 0.4 + 0.3 * (1.0 - val / n) if n > 0 else 0.4
        
        G.add_node(resource_id, type=2, weight=weight)
    
    # Connect variables to their domain resources with exponential decay
    for i in range(1, m + 1):
        for j in range(0, n + 1):
            var_id = f't_{i}_{j}'
            for val in range(0, n + 1):
                resource_id = f'domain_{val}'
                # Use exponential decay based on "distance" from natural assignment
                distance = abs(val - j) if j > 0 else (0.5 if val == 0 else 1.0)
                weight = math.exp(-2.0 * distance / n) if n > 0 else 0.5
                if weight > 0.1:  # Only add significant edges
                    G.add_edge(var_id, resource_id, weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()