#!/usr/bin/env python3
"""
Graph converter for seat-moving problem.
Created using subagent_prompt.md version: v_02

This problem is about moving people from initial seat positions to goal positions.
Key challenges: minimizing move steps while respecting swap constraints and empty seat availability.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the seat-moving problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with person-seat-constraint interactions
    - Person nodes (type 0): weighted by displacement from start to goal
    - Seat nodes (type 2): weighted by occupation/transition importance  
    - Constraint nodes (type 1): move restrictions, conflicts, goal requirements
    - Edges model movement possibilities and constraint participation
    """
    # Extract problem parameters
    S = json_data.get('S', 0)  # number of seats
    P = json_data.get('P', 0)  # number of people
    start = json_data.get('Start', [])
    goal = json_data.get('Goal', [])
    
    # Can_swap array missing from JSON - use DZN fallback if available
    can_swap = []
    if 'Can_swap' in json_data:
        can_swap = json_data['Can_swap']
    else:
        # Fallback: assume alternating pattern based on person index
        can_swap = [i % 2 == 1 for i in range(P)]
    
    G = nx.Graph()
    
    # Calculate displacement metrics for weighting
    person_positions = {}  # person_id -> (start_pos, goal_pos)
    seat_importance = [0.0] * S  # how critical each seat is for moves
    
    for seat_idx in range(S):
        if seat_idx < len(start):
            start_person = start[seat_idx]
            if start_person > 0:  # non-empty seat
                person_positions[start_person] = [seat_idx + 1, None]
        
        if seat_idx < len(goal):
            goal_person = goal[seat_idx]
            if goal_person > 0:  # non-empty seat
                if goal_person in person_positions:
                    person_positions[goal_person][1] = seat_idx + 1
                else:
                    person_positions[goal_person] = [None, seat_idx + 1]
    
    # Calculate max displacement for normalization
    max_displacement = 0
    for person_id, (start_pos, goal_pos) in person_positions.items():
        if start_pos and goal_pos:
            displacement = abs(start_pos - goal_pos)
            max_displacement = max(max_displacement, displacement)
            # Seats involved in large moves are more important
            seat_importance[start_pos - 1] += displacement
            seat_importance[goal_pos - 1] += displacement
    
    max_displacement = max(max_displacement, 1)  # avoid division by zero
    max_seat_importance = max(seat_importance) if seat_importance else 1
    
    # Add person nodes (type 0) - weighted by movement difficulty
    for person_id in range(1, P + 1):
        if person_id in person_positions:
            start_pos, goal_pos = person_positions[person_id]
            if start_pos and goal_pos:
                # Non-linear displacement weight - exponential for large moves
                displacement = abs(start_pos - goal_pos)
                base_weight = displacement / max_displacement
                # Add swap penalty - non-swappable people are harder to move
                swap_bonus = 0.3 if (person_id <= len(can_swap) and can_swap[person_id - 1]) else 0.0
                weight = min(1.0, base_weight + swap_bonus)
            else:
                weight = 0.5  # unknown position
        else:
            weight = 0.1  # person not in initial config
        
        G.add_node(f'person_{person_id}', type=0, weight=weight)
    
    # Add seat nodes (type 2) - weighted by transition importance
    empty_seats = 0
    for seat_idx in range(S):
        # Calculate importance based on start/goal occupancy and movement through it
        importance = seat_importance[seat_idx] / max_seat_importance if max_seat_importance > 0 else 0.5
        
        # Check if seat is empty in start or goal (critical for moves)
        start_person = start[seat_idx] if seat_idx < len(start) else 0
        goal_person = goal[seat_idx] if seat_idx < len(goal) else 0
        
        if start_person == 0 or goal_person == 0:
            empty_seats += 1
            importance += 0.4  # empty seats are crucial for moves
        
        # Different occupancy in start vs goal increases importance
        if start_person != goal_person:
            importance += 0.3
        
        G.add_node(f'seat_{seat_idx + 1}', type=2, weight=min(1.0, importance))
    
    # Add constraint nodes (type 1)
    
    # 1. Goal position constraints - one per person-seat goal assignment
    for person_id, (start_pos, goal_pos) in person_positions.items():
        if goal_pos:
            # Weight by difficulty of reaching goal
            displacement = abs(start_pos - goal_pos) if start_pos else S // 2
            weight = min(1.0, 0.3 + 0.7 * (displacement / max_displacement))
            G.add_node(f'goal_constraint_p{person_id}', type=1, weight=weight)
            
            # Connect person to their goal constraint
            G.add_edge(f'person_{person_id}', f'goal_constraint_p{person_id}', weight=1.0)
            # Connect goal seat to constraint
            G.add_edge(f'seat_{goal_pos}', f'goal_constraint_p{person_id}', weight=1.0)
    
    # 2. Movement constraints - model swap restrictions
    swap_enabled_count = sum(1 for can_swap_val in can_swap if can_swap_val)
    for person_id in range(1, P + 1):
        if person_id <= len(can_swap):
            if can_swap[person_id - 1]:
                # Swappable person - easier moves
                weight = 0.4
            else:
                # Non-swappable person - must use empty seats
                weight = 0.8 if empty_seats < 3 else 0.6
        else:
            weight = 0.5
        
        G.add_node(f'move_constraint_p{person_id}', type=1, weight=weight)
        G.add_edge(f'person_{person_id}', f'move_constraint_p{person_id}', weight=1.0)
    
    # 3. Spatial conflict constraints for close positions requiring swaps
    for person_id in range(1, P + 1):
        if person_id not in person_positions:
            continue
        start_pos, goal_pos = person_positions[person_id]
        if not (start_pos and goal_pos):
            continue
            
        # Find other people with overlapping movement paths
        for other_id in range(person_id + 1, P + 1):
            if other_id not in person_positions:
                continue
            other_start, other_goal = person_positions[other_id]
            if not (other_start and other_goal):
                continue
            
            # Check for crossing moves or adjacent positions
            moves_cross = (start_pos < other_start and goal_pos > other_goal) or \
                         (start_pos > other_start and goal_pos < other_goal)
            adjacent_conflict = abs(start_pos - other_start) <= 2 or abs(goal_pos - other_goal) <= 2
            
            if moves_cross or adjacent_conflict:
                # Create conflict constraint
                conflict_severity = 1.0 if moves_cross else 0.6
                G.add_node(f'conflict_p{person_id}_p{other_id}', type=1, weight=conflict_severity)
                
                # Connect both people to conflict
                G.add_edge(f'person_{person_id}', f'conflict_p{person_id}_p{other_id}', weight=0.8)
                G.add_edge(f'person_{other_id}', f'conflict_p{person_id}_p{other_id}', weight=0.8)
                
                # Connect relevant seats
                for seat in [start_pos, goal_pos, other_start, other_goal]:
                    if seat:
                        G.add_edge(f'seat_{seat}', f'conflict_p{person_id}_p{other_id}', 
                                 weight=0.5)
    
    # 4. Empty seat availability constraint
    if empty_seats > 0:
        # Weight inversely to number of empty seats - fewer empties = harder
        weight = max(0.3, 1.0 - (empty_seats / max(S - P, 1)))
        G.add_node('empty_seat_constraint', type=1, weight=weight)
        
        # Connect all empty seats and non-swappable people
        for seat_idx in range(S):
            start_person = start[seat_idx] if seat_idx < len(start) else 0
            goal_person = goal[seat_idx] if seat_idx < len(goal) else 0
            if start_person == 0 or goal_person == 0:
                G.add_edge(f'seat_{seat_idx + 1}', 'empty_seat_constraint', weight=0.7)
        
        for person_id in range(1, P + 1):
            if person_id <= len(can_swap) and not can_swap[person_id - 1]:
                G.add_edge(f'person_{person_id}', 'empty_seat_constraint', weight=0.9)
    
    # Add direct movement edges between people and seats
    for person_id, (start_pos, goal_pos) in person_positions.items():
        # Connect person to their start seat (where they currently are)
        if start_pos:
            # Connection strength based on how easy it is to leave this position
            displacement = abs(start_pos - goal_pos) if goal_pos else 1
            leave_difficulty = min(1.0, 0.2 + 0.6 * (displacement / max_displacement))
            G.add_edge(f'person_{person_id}', f'seat_{start_pos}', weight=leave_difficulty)
        
        # Connect person to their goal seat (where they need to go)
        if goal_pos:
            arrival_difficulty = 0.8  # reaching goal is generally hard
            G.add_edge(f'person_{person_id}', f'seat_{goal_pos}', weight=arrival_difficulty)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()