#!/usr/bin/env python3
"""
Graph converter for ship-schedule.cp problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling ships to depart from a port while maximizing cargo throughput.
Key challenges: tide constraints limit departure times and draft, tug resource limits, 
berth swapping requirements, and minimum separation times between ships.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the ship scheduling problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with ship nodes (type 0) and constraint nodes (type 1).
    Ships are weighted by their cargo potential and tug requirements.
    Time slots become resource nodes (type 2) with capacity-based weights.
    Constraints include separation requirements, berth swaps, tug capacity, and tide limitations.
    """
    # Extract problem parameters
    n_ships = json_data.get('NShips', 0)
    n_time_slots = json_data.get('NTimeSlots', 0)
    n_berth_swaps = json_data.get('NBerthSwaps', 0)
    n_tugs = json_data.get('NTugs', 0)
    
    earliest_start = json_data.get('EarliestStartTimeSlotForShip', [])
    tonnes_per_cm = json_data.get('TonnesPerCmDraft', [])
    min_separation = json_data.get('MinSeparationTimeSlots', [])
    max_sailing_draft = json_data.get('MaxSailingDraft_cm', [])
    tug_sets_per_ship = json_data.get('TugSetsPerShip', [])
    tug_turnaround = json_data.get('TugTurnaroundTimeSlots', [])
    
    berth_incoming = json_data.get('BerthSwap_Incoming', [])
    berth_outgoing = json_data.get('BerthSwap_Outgoing', [])
    berth_max_diff = json_data.get('BerthSwap_MaxTimeDiff', [])
    
    incoming_ships = set(json_data.get('IncomingShips', []))
    outgoing_ships = set(json_data.get('OutgoingShips', []))
    
    G = nx.Graph()
    
    # Calculate max values for normalization
    max_tonnes = max(tonnes_per_cm) if tonnes_per_cm else 1
    max_draft = max(max_sailing_draft) if max_sailing_draft else 1
    max_tug_sets = max(tug_sets_per_ship) if tug_sets_per_ship else 1
    
    # Ship nodes (type 0) - weighted by cargo potential and complexity
    for ship in range(n_ships):
        # Base weight from cargo potential
        tonnes = tonnes_per_cm[ship] if ship < len(tonnes_per_cm) else 0
        cargo_weight = tonnes / max_tonnes if max_tonnes > 0 else 0.5
        
        # Adjust by tug complexity - ships requiring more tugs are more complex
        tug_complexity = 0.5
        if ship < len(tug_sets_per_ship):
            tug_count = sum(tug_sets_per_ship[ship*2:(ship+1)*2])  # Sum across tug sets
            tug_complexity = min(1.0, tug_count / (max_tug_sets * 2))
        
        # Ships with later earliest start times are more constrained
        time_constraint = 0.5
        if ship < len(earliest_start):
            time_constraint = earliest_start[ship] / n_time_slots if n_time_slots > 0 else 0.5
        
        # Combine factors with non-linear weighting
        ship_weight = (0.4 * cargo_weight + 0.3 * tug_complexity + 0.3 * time_constraint)
        ship_weight = min(1.0, ship_weight)
        
        G.add_node(f'ship_{ship}', type=0, weight=ship_weight)
    
    # Time slot resource nodes (type 2) - weighted by available draft capacity
    for t in range(n_time_slots):
        # Calculate total draft capacity available across all ships at this time
        total_capacity = 0
        valid_drafts = []
        
        for ship in range(n_ships):
            draft_idx = t * n_ships + ship
            if draft_idx < len(max_sailing_draft):
                draft = max_sailing_draft[draft_idx]
                if draft > 0:  # Ship can sail at this time
                    total_capacity += draft
                    valid_drafts.append(draft)
        
        # Weight by relative capacity and variation
        if valid_drafts:
            avg_capacity = total_capacity / len(valid_drafts)
            capacity_weight = min(1.0, avg_capacity / max_draft) if max_draft > 0 else 0.5
            # Boost weight for time slots with high variation (more strategic importance)
            if len(valid_drafts) > 1:
                variance = sum((d - avg_capacity) ** 2 for d in valid_drafts) / len(valid_drafts)
                variation_boost = min(0.3, math.sqrt(variance) / max_draft) if max_draft > 0 else 0
                capacity_weight = min(1.0, capacity_weight + variation_boost)
        else:
            capacity_weight = 0.1  # Very low weight for unusable time slots
        
        G.add_node(f'time_{t}', type=2, weight=capacity_weight)
    
    # Separation constraint nodes (type 1) - one per ship pair
    separation_matrix = []
    for i in range(n_ships):
        row = []
        for j in range(n_ships):
            idx = i * n_ships + j
            sep = min_separation[idx] if idx < len(min_separation) else 0
            row.append(sep)
        separation_matrix.append(row)
    
    max_separation = max(max(row) for row in separation_matrix) if separation_matrix else 1
    
    for ship1 in range(n_ships):
        for ship2 in range(ship1 + 1, n_ships):
            sep1 = separation_matrix[ship1][ship2]
            sep2 = separation_matrix[ship2][ship1]
            max_sep = max(sep1, sep2)
            
            if max_sep > 0:  # Only create constraint if separation is required
                # Weight by tightness of separation requirement
                separation_weight = max_sep / max_separation if max_separation > 0 else 0.5
                # Use non-linear scaling to emphasize very tight constraints
                separation_weight = math.pow(separation_weight, 0.7)
                
                constraint_id = f'separation_{ship1}_{ship2}'
                G.add_node(constraint_id, type=1, weight=separation_weight)
                
                # Connect ships to separation constraint
                G.add_edge(f'ship_{ship1}', constraint_id, weight=sep1 / max_separation if max_separation > 0 else 0.5)
                G.add_edge(f'ship_{ship2}', constraint_id, weight=sep2 / max_separation if max_separation > 0 else 0.5)
    
    # Berth swap constraint nodes (type 1)
    for swap in range(n_berth_swaps):
        if (swap < len(berth_incoming) and swap < len(berth_outgoing) and 
            swap < len(berth_max_diff)):
            
            incoming = berth_incoming[swap] - 1  # Convert to 0-indexed
            outgoing = berth_outgoing[swap] - 1
            max_diff = berth_max_diff[swap]
            
            # Weight by tightness of timing constraint
            time_constraint_weight = 1.0 - (max_diff / n_time_slots) if n_time_slots > 0 else 0.8
            time_constraint_weight = max(0.1, time_constraint_weight)
            
            constraint_id = f'berth_swap_{swap}'
            G.add_node(constraint_id, type=1, weight=time_constraint_weight)
            
            # Connect ships involved in berth swap
            if 0 <= incoming < n_ships:
                G.add_edge(f'ship_{incoming}', constraint_id, weight=0.9)
            if 0 <= outgoing < n_ships:
                G.add_edge(f'ship_{outgoing}', constraint_id, weight=0.9)
    
    # Tug capacity constraint nodes (type 1) - one per time slot
    for t in range(n_time_slots):
        # Calculate potential tug demand at this time slot
        max_demand = 0
        for ship in range(n_ships):
            if ship < len(tug_sets_per_ship):
                ship_tug_demand = sum(tug_sets_per_ship[ship*2:(ship+1)*2])
                max_demand += ship_tug_demand
        
        # Weight by potential oversubscription
        if max_demand > 0:
            oversubscription = max_demand / n_tugs if n_tugs > 0 else 1.0
            tug_weight = min(1.0, oversubscription * 0.8)
            # Use exponential scaling for high oversubscription
            if oversubscription > 1.5:
                tug_weight = min(1.0, 0.8 + 0.2 * math.exp(oversubscription - 1.5))
        else:
            tug_weight = 0.3
        
        G.add_node(f'tug_capacity_{t}', type=1, weight=tug_weight)
    
    # Ship-time slot participation edges
    for ship in range(n_ships):
        for t in range(n_time_slots):
            draft_idx = t * n_ships + ship
            if draft_idx < len(max_sailing_draft):
                draft = max_sailing_draft[draft_idx]
                if draft > 0:  # Ship can sail at this time
                    # Edge weight based on draft availability and cargo potential
                    draft_ratio = draft / max_draft if max_draft > 0 else 0.5
                    tonnes = tonnes_per_cm[ship] if ship < len(tonnes_per_cm) else 0
                    cargo_potential = (draft_ratio * tonnes / max_tonnes) if max_tonnes > 0 else draft_ratio
                    
                    # Early constraint check
                    can_start = True
                    if ship < len(earliest_start) and t < earliest_start[ship] - 1:
                        can_start = False
                    
                    edge_weight = cargo_potential if can_start else cargo_potential * 0.3
                    edge_weight = min(1.0, edge_weight)
                    
                    G.add_edge(f'ship_{ship}', f'time_{t}', weight=edge_weight)
                    
                    # Connect ship to tug capacity constraint for this time
                    G.add_edge(f'ship_{ship}', f'tug_capacity_{t}', weight=0.6)
    
    # Add global complexity constraint node if problem is large enough
    if n_ships >= 5:
        total_complexity = n_ships * n_time_slots * n_berth_swaps
        complexity_weight = min(1.0, math.log(total_complexity) / 10.0)
        
        G.add_node('global_complexity', type=1, weight=complexity_weight)
        
        # Connect most constrained ships to global complexity
        ship_complexities = []
        for ship in range(n_ships):
            complexity = 0
            if ship < len(tug_sets_per_ship):
                complexity += sum(tug_sets_per_ship[ship*2:(ship+1)*2])
            if ship < len(earliest_start):
                complexity += earliest_start[ship] / n_time_slots
            if ship + 1 in incoming_ships or ship + 1 in outgoing_ships:
                complexity += 0.5
            ship_complexities.append((ship, complexity))
        
        # Connect top 3 most complex ships
        ship_complexities.sort(key=lambda x: x[1], reverse=True)
        for i, (ship, complexity) in enumerate(ship_complexities[:3]):
            weight = 0.8 - i * 0.2  # Decreasing weights
            G.add_edge(f'ship_{ship}', 'global_complexity', weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()