#!/usr/bin/env python3
"""
Graph converter for ship-schedule.mip problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling ships to sail at optimal times while maximizing total tonnage.
Key challenges: tide-dependent draft limits, ship separation requirements, berth swaps, tug availability.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the ship scheduling problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with ships as variables and various constraints
    - Ships are decision variables (type 0) weighted by tonnage potential
    - Time slot constraints limit when ships can sail (type 1)
    - Separation constraints enforce minimum time between ships (type 1)
    - Berth swap constraints coordinate berth usage (type 1)
    - Tug capacity constraints limit concurrent tug usage (type 1)
    """
    # Extract problem parameters
    n_ships = json_data.get('NShips', 0)
    n_time_slots = json_data.get('NTimeSlots', 0)
    n_berth_swaps = json_data.get('NBerthSwaps', 0)
    n_tugs = json_data.get('NTugs', 0)
    
    earliest_start = json_data.get('EarliestStartTimeSlotForShip', [])
    tonnes_per_cm = json_data.get('TonnesPerCmDraft', [])
    min_separation = json_data.get('MinSeparationTimeSlots', [])
    max_sailing_draft = json_data.get('MaxSailingDraft_cm', [])
    
    berth_incoming = json_data.get('BerthSwap_Incoming', [])
    berth_outgoing = json_data.get('BerthSwap_Outgoing', [])
    berth_max_time_diff = json_data.get('BerthSwap_MaxTimeDiff', [])
    
    tug_sets_per_ship = json_data.get('TugSetsPerShip', [])
    tug_turnaround = json_data.get('TugTurnaroundTimeSlots', [])
    
    G = nx.Graph()
    
    # Add ship nodes (decision variables - type 0)
    # Weight by tonnage potential and scheduling flexibility
    max_tonnes = max(tonnes_per_cm) if tonnes_per_cm else 1
    max_draft_values = []
    
    # Calculate max possible draft for each ship across all time slots
    for ship in range(n_ships):
        max_draft_for_ship = 0
        for t in range(n_time_slots):
            if t < len(max_sailing_draft):
                draft_idx = t * n_ships + ship
                if draft_idx < len(max_sailing_draft):
                    max_draft_for_ship = max(max_draft_for_ship, max_sailing_draft[draft_idx])
        max_draft_values.append(max_draft_for_ship)
    
    max_draft_overall = max(max_draft_values) if max_draft_values else 1
    
    for ship in range(n_ships):
        # Weight combines tonnage potential and maximum achievable draft
        tonnage_weight = tonnes_per_cm[ship] / max_tonnes if ship < len(tonnes_per_cm) else 0.5
        draft_weight = max_draft_values[ship] / max_draft_overall if max_draft_overall > 0 else 0.5
        
        # Use non-linear combination to emphasize high-value ships
        ship_weight = math.sqrt(tonnage_weight * draft_weight)
        
        G.add_node(f'ship_{ship}', type=0, weight=ship_weight)
    
    # Add time slot availability constraints (type 1)
    # Weight by how restrictive the time slot is
    for t in range(n_time_slots):
        available_ships = 0
        total_draft_available = 0
        
        for ship in range(n_ships):
            draft_idx = t * n_ships + ship
            if draft_idx < len(max_sailing_draft) and max_sailing_draft[draft_idx] > 0:
                available_ships += 1
                total_draft_available += max_sailing_draft[draft_idx]
        
        # Weight by scarcity - fewer available ships means higher constraint weight
        scarcity = 1.0 - (available_ships / n_ships) if n_ships > 0 else 0.5
        # Use exponential to emphasize very restrictive time slots
        constraint_weight = 1.0 - math.exp(-3.0 * scarcity)
        
        constraint_id = f'timeslot_{t}'
        G.add_node(constraint_id, type=1, weight=constraint_weight)
        
        # Connect ships to time slots they can use
        for ship in range(n_ships):
            draft_idx = t * n_ships + ship
            if (draft_idx < len(max_sailing_draft) and 
                max_sailing_draft[draft_idx] > 0 and
                t >= (earliest_start[ship] - 1 if ship < len(earliest_start) else 0)):
                
                # Edge weight based on draft quality at this time
                draft_quality = max_sailing_draft[draft_idx] / max_draft_overall if max_draft_overall > 0 else 0.5
                G.add_edge(f'ship_{ship}', constraint_id, weight=draft_quality)
    
    # Add separation constraints between ships (type 1)
    # Weight by separation requirement and ship importance
    ship_idx = 0
    for ship1 in range(n_ships):
        for ship2 in range(ship1 + 1, n_ships):
            if ship_idx < len(min_separation):
                sep_time = min_separation[ship_idx]
                if sep_time > 0:
                    # Weight by separation requirement relative to problem scale
                    sep_weight = min(1.0, sep_time / (n_time_slots * 0.1))
                    # Use square root for non-linear scaling
                    sep_weight = math.sqrt(sep_weight)
                    
                    constraint_id = f'separation_{ship1}_{ship2}'
                    G.add_node(constraint_id, type=1, weight=sep_weight)
                    
                    # Connect both ships to this separation constraint
                    G.add_edge(f'ship_{ship1}', constraint_id, weight=0.8)
                    G.add_edge(f'ship_{ship2}', constraint_id, weight=0.8)
                
                ship_idx += 1
    
    # Add berth swap constraints (type 1)
    for swap in range(n_berth_swaps):
        if (swap < len(berth_incoming) and swap < len(berth_outgoing) and 
            swap < len(berth_max_time_diff)):
            
            incoming_ship = berth_incoming[swap] - 1  # Convert to 0-based
            outgoing_ship = berth_outgoing[swap] - 1
            max_time_diff = berth_max_time_diff[swap]
            
            if 0 <= incoming_ship < n_ships and 0 <= outgoing_ship < n_ships:
                # Weight by time constraint tightness
                time_weight = 1.0 - (max_time_diff / n_time_slots) if n_time_slots > 0 else 0.8
                
                constraint_id = f'berth_swap_{swap}'
                G.add_node(constraint_id, type=1, weight=time_weight)
                
                # Connect ships involved in berth swap
                G.add_edge(f'ship_{incoming_ship}', constraint_id, weight=0.9)
                G.add_edge(f'ship_{outgoing_ship}', constraint_id, weight=0.9)
    
    # Add tug capacity constraints (type 1)
    # Create constraints based on tug requirements and turnaround times
    total_tug_demand = 0
    max_turnaround = 0
    
    for ship in range(n_ships):
        if ship < len(tug_sets_per_ship):
            # Sum tug requirements for this ship
            ship_tug_demand = 0
            ship_idx_base = ship * 2  # Each ship has up to 2 tug sets
            for tug_set in range(2):  # MaxNTugSets = 2
                tug_idx = ship_idx_base + tug_set
                if tug_idx < len(tug_sets_per_ship):
                    ship_tug_demand += tug_sets_per_ship[tug_idx]
                if tug_idx < len(tug_turnaround):
                    max_turnaround = max(max_turnaround, tug_turnaround[tug_idx])
            
            total_tug_demand += ship_tug_demand
    
    # Create tug capacity constraint
    if n_tugs > 0 and total_tug_demand > 0:
        # Weight by capacity pressure
        capacity_pressure = min(1.0, total_tug_demand / n_tugs)
        # Use exponential scaling for high pressure situations
        tug_weight = 1.0 - math.exp(-2.0 * capacity_pressure)
        
        tug_constraint_id = 'tug_capacity'
        G.add_node(tug_constraint_id, type=1, weight=tug_weight)
        
        # Connect all ships that need tugs
        for ship in range(n_ships):
            ship_tug_demand = 0
            ship_idx_base = ship * 2
            for tug_set in range(2):
                tug_idx = ship_idx_base + tug_set
                if tug_idx < len(tug_sets_per_ship):
                    ship_tug_demand += tug_sets_per_ship[tug_idx]
            
            if ship_tug_demand > 0:
                # Edge weight by tug demand intensity
                demand_weight = min(1.0, ship_tug_demand / n_tugs)
                G.add_edge(f'ship_{ship}', tug_constraint_id, weight=demand_weight)
    
    # Add global complexity constraint (optional)
    # Represents overall problem difficulty
    complexity_factors = [
        n_ships / 10.0,  # Ship count
        n_berth_swaps / 5.0,  # Berth swap complexity
        (total_tug_demand / n_tugs) if n_tugs > 0 else 0.5,  # Tug pressure
        len([sep for sep in min_separation if sep > 0]) / (n_ships * (n_ships - 1) / 2)  # Separation density
    ]
    
    overall_complexity = min(1.0, sum(complexity_factors) / 4.0)
    complexity_weight = math.sqrt(overall_complexity)
    
    G.add_node('global_complexity', type=1, weight=complexity_weight)
    
    # Connect high-value ships to global complexity
    for ship in range(n_ships):
        ship_node = f'ship_{ship}'
        if ship_node in G.nodes:
            ship_weight = G.nodes[ship_node]['weight']
            if ship_weight > 0.7:  # Only connect important ships
                G.add_edge(ship_node, 'global_complexity', weight=ship_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()