#!/usr/bin/env python3
"""
Graph converter for slow_convergence problem.
Created using subagent_prompt.md version: v_02

This problem is about constraint satisfaction with two ordered arrays x and y, 
designed to test slow convergence in constraint propagation.
Key challenges: tight interactions between ordering constraints and bounds,
propagation dependencies between arrays, scale-dependent difficulty.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the slow_convergence problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the constraint propagation structure and variable dependencies
    - Variables: x[0..n] and y[0..n] positions with position-based weights
    - Constraints: Ordering constraints, bound constraints, linking constraint
    - Key insight: Position in arrays affects constraint tightness and propagation difficulty
    """
    n = json_data.get('n', 10)
    
    G = nx.Graph()
    
    # Variable nodes for y array (type 0)
    # Position-based weights: middle positions are more constrained
    for i in range(n + 1):
        # Weight based on position centrality and constraint involvement
        if i == 0:
            # y[0] is most constrained (lower bound + affects all upper bounds)
            weight = 1.0
        elif i == n:
            # y[n] links to x[0] - special role
            weight = 0.8
        else:
            # Middle positions more constrained than edges
            centrality = 1.0 - abs(i - n/2) / (n/2)
            weight = 0.3 + 0.4 * centrality
        
        G.add_node(f'y_{i}', type=0, weight=weight)
    
    # Variable nodes for x array (type 0)
    for i in range(n + 1):
        if i == 0:
            # x[0] receives constraint from y[n]
            weight = 0.7
        else:
            # Ordering constraints - later positions less constrained
            weight = 0.4 + 0.3 * math.exp(-i / (n/3))
        
        G.add_node(f'x_{i}', type=0, weight=weight)
    
    # Constraint nodes (type 1)
    
    # Y ordering constraints: y[i-1] <= y[i]
    for i in range(2, n + 1):
        # Later constraints might be easier to satisfy
        tightness = 0.6 + 0.3 * math.exp(-i / (n/4))
        G.add_node(f'y_order_{i}', type=1, weight=tightness)
    
    # Y upper bound constraints: y[0] - y[i] <= n - i + 1
    for i in range(1, n + 1):
        # Tightness increases for later positions (smaller bounds)
        bound_slack = (n - i + 1) / n
        tightness = 1.0 - bound_slack
        G.add_node(f'y_bound_{i}', type=1, weight=tightness)
    
    # X ordering constraints: x[i] <= x[j] for i < j
    constraint_count = 0
    for i in range(1, n):
        for j in range(i + 1, n + 1):
            # Shorter spans are tighter constraints
            span = j - i
            tightness = math.exp(-span / (n/3))
            G.add_node(f'x_order_{i}_{j}', type=1, weight=tightness)
            constraint_count += 1
    
    # Linking constraint: y[n] <= x[0]
    G.add_node('link_constraint', type=1, weight=0.9)
    
    # Lower bound constraint: y[0] >= n
    G.add_node('y_lower_bound', type=1, weight=0.8)
    
    # Edges: Variable-Constraint participation (bipartite)
    
    # Y ordering constraint edges
    for i in range(2, n + 1):
        G.add_edge(f'y_{i-1}', f'y_order_{i}', weight=1.0)
        G.add_edge(f'y_{i}', f'y_order_{i}', weight=1.0)
    
    # Y upper bound constraint edges
    for i in range(1, n + 1):
        G.add_edge(f'y_0', f'y_bound_{i}', weight=1.0)
        G.add_edge(f'y_{i}', f'y_bound_{i}', weight=1.0)
    
    # X ordering constraint edges
    for i in range(1, n):
        for j in range(i + 1, n + 1):
            span = j - i
            # Weight by constraint scope impact
            edge_weight = min(1.0, 2.0 / span)
            G.add_edge(f'x_{i}', f'x_order_{i}_{j}', weight=edge_weight)
            G.add_edge(f'x_{j}', f'x_order_{i}_{j}', weight=edge_weight)
    
    # Linking constraint edges
    G.add_edge(f'y_{n}', 'link_constraint', weight=1.0)
    G.add_edge('x_0', 'link_constraint', weight=1.0)
    
    # Lower bound constraint edge
    G.add_edge('y_0', 'y_lower_bound', weight=1.0)
    
    # Add critical propagation edges (variable-to-variable for tight dependencies)
    # y[0] affects all other y variables through bound constraints
    for i in range(1, min(n + 1, 6)):  # Limit to avoid too many edges
        propagation_strength = math.exp(-i / 3.0)
        if propagation_strength > 0.1:
            G.add_edge('y_0', f'y_{i}', weight=propagation_strength)
    
    # Connection between y[n] and x[0] due to linking constraint
    G.add_edge(f'y_{n}', 'x_0', weight=0.8)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()