#!/usr/bin/env python3
"""
Graph converter for smelt (smelting/production scheduling) problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling production orders with different recipes on limited
production lines, subject to mineral flow constraints and production rules.
Key challenges: resource contention, complex precedence constraints, recipe-order interactions.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the smelting scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the scheduling bottlenecks and constraints
    - Orders as variable nodes (what we're scheduling)
    - Recipes as resource nodes (production patterns)
    - Production lines as resource nodes (machines)
    - Minerals as resource nodes (flow-limited resources)
    - Production rules as constraint nodes (precedence/timing)
    - Resource capacity constraints as constraint nodes
    """
    # Extract problem data
    m = json_data.get('m', 0)  # number of minerals
    r = json_data.get('r', 0)  # number of recipes
    n = json_data.get('n', 0)  # number of orders
    nl = json_data.get('nl', 0)  # number of production lines
    p = json_data.get('p', 0)  # number of production rules
    
    f = json_data.get('f', [])  # mineral flow rates
    t = json_data.get('t', [])  # recipe for each order
    h = json_data.get('h', [])  # height of each order
    w = json_data.get('w', [])  # width of each order
    c = json_data.get('c', [])  # mineral consumption matrix [recipe][mineral]
    
    typ = json_data.get('typ', [])  # production rule types
    k1 = json_data.get('k1', [])   # first recipe in rule
    k2 = json_data.get('k2', [])   # second recipe in rule
    d = json_data.get('d', [])     # delay in rule
    
    G = nx.Graph()
    
    # 1. ORDER NODES (Type 0 - variables we're scheduling)
    # Weight by processing size (h*w) - larger orders are more constraining
    max_size = max([h[i]*w[i] for i in range(n)]) if n > 0 else 1
    for i in range(n):
        order_size = h[i] * w[i]
        # Larger orders are more critical for scheduling
        weight = math.sqrt(order_size / max_size)  # Non-linear scaling
        G.add_node(f'order_{i}', type=0, weight=weight)
    
    # 2. RECIPE NODES (Type 2 - shared production patterns)
    # Weight by total mineral consumption and usage frequency
    total_consumption = {}
    recipe_usage = [0] * r
    
    # Calculate recipe complexity and usage
    for recipe in range(r):
        total_mineral_use = 0
        for mineral in range(m):
            idx = recipe * m + mineral
            if idx < len(c):
                total_mineral_use += c[idx]
        total_consumption[recipe] = total_mineral_use
        
        # Count how many orders use this recipe
        for i in range(n):
            if i < len(t) and t[i] == recipe + 1:  # recipes are 1-indexed
                recipe_usage[recipe] += 1
    
    max_consumption = max(total_consumption.values()) if total_consumption else 1
    max_usage = max(recipe_usage) if recipe_usage else 1
    
    for recipe in range(r):
        # Weight by complexity (mineral use) and demand (usage frequency)
        complexity = total_consumption.get(recipe, 0) / max_consumption
        demand = recipe_usage[recipe] / max_usage
        weight = (complexity + demand) / 2
        G.add_node(f'recipe_{recipe}', type=2, weight=weight)
    
    # 3. MINERAL NODES (Type 2 - flow-limited resources)
    # Weight by scarcity (inverse of flow rate)
    max_flow = max(f) if f else 1
    for mineral in range(m):
        flow_rate = f[mineral] if mineral < len(f) else 1
        # Lower flow rate = higher scarcity = higher weight
        scarcity = 1.0 - (flow_rate / max_flow)
        weight = math.exp(scarcity) / math.exp(1.0)  # Exponential scaling for scarcity
        G.add_node(f'mineral_{mineral}', type=2, weight=weight)
    
    # 4. PRODUCTION LINE NODES (Type 2 - machine resources)
    # All lines have equal capacity, weight by utilization potential
    line_utilization = nl / max(r, 1)  # Simple utilization estimate
    for line in range(nl):
        weight = min(line_utilization, 1.0)
        G.add_node(f'line_{line}', type=2, weight=weight)
    
    # 5. PRODUCTION RULE CONSTRAINT NODES (Type 1)
    # Weight by delay impact and rule type complexity
    max_delay = max(d) if d else 1
    for rule in range(p):
        if rule < len(d) and rule < len(typ):
            delay = d[rule]
            rule_type = typ[rule]
            
            # Different rule types have different complexity
            type_complexity = {1: 0.8, 2: 0.9, 3: 0.7, 4: 0.6}
            complexity = type_complexity.get(rule_type, 0.5)
            
            # Longer delays are more constraining
            delay_impact = delay / max_delay if max_delay > 0 else 0.5
            
            weight = (complexity + delay_impact) / 2
            G.add_node(f'rule_{rule}', type=1, weight=weight)
    
    # 6. RESOURCE CAPACITY CONSTRAINT NODES (Type 1)
    # One for each mineral flow constraint
    for mineral in range(m):
        # Calculate total demand for this mineral
        total_demand = 0
        for recipe in range(r):
            recipe_demand = 0
            idx = recipe * m + mineral
            if idx < len(c):
                recipe_demand = c[idx]
            
            # Multiply by orders using this recipe
            for i in range(n):
                if i < len(t) and t[i] == recipe + 1:
                    recipe_demand += h[i] * w[i] if i < len(h) and i < len(w) else 1
            total_demand += recipe_demand
        
        # Constraint tightness
        flow_capacity = f[mineral] if mineral < len(f) else 1
        tightness = min(total_demand / max(flow_capacity, 1), 1.0)
        G.add_node(f'mineral_constraint_{mineral}', type=1, weight=tightness)
    
    # EDGES: Model relationships and dependencies
    
    # Order-Recipe edges (orders consume recipe patterns)
    for i in range(n):
        if i < len(t):
            recipe_id = t[i] - 1  # Convert from 1-indexed
            if 0 <= recipe_id < r:
                order_size = h[i] * w[i] if i < len(h) and i < len(w) else 1
                weight = min(order_size / max_size, 1.0)
                G.add_edge(f'order_{i}', f'recipe_{recipe_id}', weight=weight)
    
    # Recipe-Mineral edges (recipes consume minerals)
    for recipe in range(r):
        for mineral in range(m):
            idx = recipe * m + mineral
            if idx < len(c) and c[idx] > 0:
                consumption = c[idx]
                flow_capacity = f[mineral] if mineral < len(f) else 1
                # Weight by consumption relative to capacity
                weight = min(consumption / max(flow_capacity, 1), 1.0)
                G.add_edge(f'recipe_{recipe}', f'mineral_{mineral}', weight=weight)
                
                # Also connect to mineral constraint
                G.add_edge(f'recipe_{recipe}', f'mineral_constraint_{mineral}', weight=weight)
    
    # Production rule edges (rules constrain recipe pairs)
    for rule in range(p):
        if (rule < len(k1) and rule < len(k2) and 
            rule < len(typ) and rule < len(d)):
            
            recipe1 = k1[rule] - 1  # Convert from 1-indexed
            recipe2 = k2[rule] - 1
            delay = d[rule]
            
            if 0 <= recipe1 < r and 0 <= recipe2 < r:
                # Weight by delay impact
                delay_weight = delay / max_delay if max_delay > 0 else 0.5
                
                G.add_edge(f'rule_{rule}', f'recipe_{recipe1}', weight=delay_weight)
                G.add_edge(f'rule_{rule}', f'recipe_{recipe2}', weight=delay_weight)
    
    # Order competition edges (orders using same recipe compete)
    recipe_orders = {}
    for i in range(n):
        if i < len(t):
            recipe_id = t[i] - 1
            if recipe_id not in recipe_orders:
                recipe_orders[recipe_id] = []
            recipe_orders[recipe_id].append(i)
    
    # Add competition edges between orders using same recipe
    for recipe_id, order_list in recipe_orders.items():
        if len(order_list) > 1:
            # Sort by size to identify major competitors
            order_sizes = [(i, h[i]*w[i] if i < len(h) and i < len(w) else 1) 
                          for i in order_list]
            order_sizes.sort(key=lambda x: x[1], reverse=True)
            
            # Connect top competitors
            for idx1 in range(min(len(order_sizes), 3)):
                for idx2 in range(idx1+1, min(len(order_sizes), 3)):
                    i1, size1 = order_sizes[idx1]
                    i2, size2 = order_sizes[idx2]
                    
                    # Competition strength based on combined size
                    competition = (size1 + size2) / (2 * max_size)
                    G.add_edge(f'order_{i1}', f'order_{i2}', weight=competition)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()