#!/usr/bin/env python3
"""
Graph converter for SPOT5 earth observation satellite problem.
Created using subagent_prompt.md version: v_02

This problem is about selecting which photographs to take from a satellite,
subject to timing and resource constraints between photographs.
Key challenges: optimization with complex binary constraints between photographs,
varying photograph importance (costs), and resource limitations.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the SPOT5 satellite observation problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with photographs as variables (type 0)
    and constraints as explicit constraint nodes (type 1).
    - Photograph nodes weighted by importance (cost values)
    - Constraint nodes weighted by scope and tightness
    - Edges represent participation in constraints
    - Add conflict edges between highly constrained photographs
    """
    
    # Extract problem data
    num_variables = json_data.get('num_variables', 0)
    costs = json_data.get('costs', [])
    num_constraints2 = json_data.get('num_constraints2', 0)
    scopes2x = json_data.get('scopes2x', [])
    scopes2y = json_data.get('scopes2y', [])
    constraints2 = json_data.get('constraints2', [])
    num_tuples2 = json_data.get('num_tuples2', [])
    cum_tuples2 = json_data.get('cum_tuples2', [])
    
    G = nx.Graph()
    
    # Variable nodes (photographs) with importance-based weights
    max_cost = max(costs) if costs else 1
    min_cost = min(costs) if costs else 0
    cost_range = max_cost - min_cost if max_cost > min_cost else 1
    
    for i in range(num_variables):
        if i < len(costs):
            # Normalize cost to [0,1] using non-linear scaling
            # Higher cost = more important = higher weight
            normalized_cost = (costs[i] - min_cost) / cost_range
            # Apply logarithmic scaling to emphasize high-value photographs
            weight = math.pow(normalized_cost, 0.7) if normalized_cost > 0 else 0.1
        else:
            weight = 0.1
        
        G.add_node(f'photo_{i+1}', type=0, weight=min(weight, 1.0))
    
    # Constraint nodes (binary constraints) with tightness-based weights
    for j in range(num_constraints2):
        if j < len(num_tuples2):
            num_tuples = num_tuples2[j]
            # Calculate constraint tightness based on number of allowed tuples
            # Fewer allowed tuples = tighter constraint = higher weight
            # Domain size is roughly (max_domain - min_domain + 1)^2 for binary constraints
            max_domain = json_data.get('max_domain', 13)
            min_domain = json_data.get('min_domain', 0)
            domain_size = max_domain - min_domain + 1
            max_possible_tuples = domain_size * domain_size
            
            if max_possible_tuples > 0:
                tightness = 1.0 - (num_tuples / max_possible_tuples)
                # Apply exponential scaling to emphasize very tight constraints
                weight = math.pow(tightness, 0.5) if tightness > 0 else 0.1
            else:
                weight = 0.5
        else:
            weight = 0.5
            
        G.add_node(f'constraint_{j+1}', type=1, weight=min(weight, 1.0))
    
    # Bipartite edges: photograph-constraint participation
    for j in range(num_constraints2):
        if j < len(scopes2x) and j < len(scopes2y):
            var1 = scopes2x[j]
            var2 = scopes2y[j]
            
            if 1 <= var1 <= num_variables and 1 <= var2 <= num_variables:
                # Edge weights based on constraint participation
                # Tighter constraints get higher edge weights
                if j < len(num_tuples2):
                    num_tuples = num_tuples2[j]
                    max_domain = json_data.get('max_domain', 13)
                    min_domain = json_data.get('min_domain', 0)
                    domain_size = max_domain - min_domain + 1
                    max_possible_tuples = domain_size * domain_size
                    
                    if max_possible_tuples > 0:
                        participation_weight = 1.0 - (num_tuples / max_possible_tuples)
                        participation_weight = min(max(participation_weight, 0.1), 1.0)
                    else:
                        participation_weight = 0.5
                else:
                    participation_weight = 0.5
                
                G.add_edge(f'photo_{var1}', f'constraint_{j+1}', weight=participation_weight)
                G.add_edge(f'photo_{var2}', f'constraint_{j+1}', weight=participation_weight)
    
    # Add conflict edges between high-value photographs that share many constraints
    # Build constraint participation map
    photo_constraints = {}
    for j in range(num_constraints2):
        if j < len(scopes2x) and j < len(scopes2y):
            var1 = scopes2x[j]
            var2 = scopes2y[j]
            
            if 1 <= var1 <= num_variables:
                if var1 not in photo_constraints:
                    photo_constraints[var1] = []
                photo_constraints[var1].append(j)
                
            if 1 <= var2 <= num_variables:
                if var2 not in photo_constraints:
                    photo_constraints[var2] = []
                photo_constraints[var2].append(j)
    
    # Add conflict edges between photographs that share many constraints
    # Focus on high-value photographs
    high_value_photos = []
    if costs:
        avg_cost = sum(costs) / len(costs)
        for i in range(num_variables):
            if i < len(costs) and costs[i] > avg_cost * 1.5:
                high_value_photos.append(i + 1)
    
    for i, photo1 in enumerate(high_value_photos):
        for photo2 in high_value_photos[i+1:]:
            if photo1 in photo_constraints and photo2 in photo_constraints:
                shared_constraints = len(set(photo_constraints[photo1]) & set(photo_constraints[photo2]))
                total_constraints = len(set(photo_constraints[photo1]) | set(photo_constraints[photo2]))
                
                if total_constraints > 0 and shared_constraints >= 2:
                    # Conflict strength based on shared constraints and photograph values
                    conflict_strength = shared_constraints / total_constraints
                    
                    # Weight by photograph importance
                    photo1_cost = costs[photo1-1] if photo1-1 < len(costs) else 1
                    photo2_cost = costs[photo2-1] if photo2-1 < len(costs) else 1
                    importance_factor = math.sqrt(photo1_cost * photo2_cost) / max_cost if max_cost > 0 else 0.5
                    
                    final_weight = min(conflict_strength * importance_factor, 1.0)
                    if final_weight > 0.3:  # Only add significant conflicts
                        G.add_edge(f'photo_{photo1}', f'photo_{photo2}', weight=final_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()