#!/usr/bin/env python3
"""
Graph converter for stable-goods-solution problem.
Created using subagent_prompt.md version: v_02

This problem is about allocating limited goods to people with preferences such that 
the allocation is stable (no person prefers someone else's allocation and can get it).
Key challenges: Preference ordering, scarcity, stability constraints, value optimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the stable goods allocation problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with people (type 0), goods (type 2), 
    and constraints (type 1). Focus on preference conflicts and scarcity.
    - People nodes: weighted by preference complexity
    - Good nodes: weighted by scarcity (demand vs availability)
    - Constraint nodes: stability constraints, capacity constraints
    - Edges model preference relationships and constraint participation
    """
    # Extract data
    m = json_data.get('m', 0)  # number of people
    n = json_data.get('n', 0)  # number of goods
    available = json_data.get('available', [])
    value = json_data.get('value', [])
    npref = json_data.get('npref', [])
    good_pref = json_data.get('good_pref', [])
    req_pref = json_data.get('req_pref', [])
    
    G = nx.Graph()
    
    # Helper function to get preference index
    def get_pref_start(person):
        return sum(npref[:person]) if person > 0 else 0
    
    # Calculate demand for each good to assess scarcity
    good_demand = [0] * n
    for person in range(m):
        pref_start = get_pref_start(person)
        for i in range(npref[person]):
            pref_idx = pref_start + i
            if pref_idx < len(good_pref):
                good_id = good_pref[pref_idx] - 1  # Convert to 0-indexed
                if 0 <= good_id < n and pref_idx < len(req_pref):
                    good_demand[good_id] += req_pref[pref_idx]
    
    # Person nodes (type 0) - weighted by preference complexity
    max_prefs = max(npref) if npref else 1
    for person in range(m):
        # Weight by preference complexity and diversity
        complexity = npref[person] / max_prefs if max_prefs > 0 else 0.5
        
        # Calculate preference spread (how many different goods they want)
        pref_start = get_pref_start(person)
        unique_goods = set()
        for i in range(npref[person]):
            pref_idx = pref_start + i
            if pref_idx < len(good_pref):
                unique_goods.add(good_pref[pref_idx])
        
        diversity = len(unique_goods) / n if n > 0 else 0.5
        weight = (complexity + diversity) / 2
        G.add_node(f'person_{person}', type=0, weight=weight)
    
    # Good nodes (type 2) - weighted by scarcity and value
    max_value = max(value) if value else 1
    max_avail = max(available) if available else 1
    for good in range(n):
        avail = available[good] if good < len(available) else 0
        good_val = value[good] if good < len(value) else 0
        demand = good_demand[good]
        
        # Scarcity: how much demand exceeds availability
        scarcity = min(demand / max(avail, 1), 3.0) / 3.0 if avail > 0 else 1.0
        
        # Value importance
        val_importance = good_val / max_value if max_value > 0 else 0.5
        
        # Combined weight using non-linear scaling
        weight = (scarcity * 0.7 + val_importance * 0.3)
        weight = min(1.0 - math.exp(-3.0 * weight), 1.0)  # Non-linear scaling
        
        G.add_node(f'good_{good}', type=2, weight=weight)
    
    # Capacity constraint nodes (type 1) - one per good
    for good in range(n):
        avail = available[good] if good < len(available) else 0
        demand = good_demand[good]
        
        # Tightness: how close demand is to capacity
        if avail > 0:
            tightness = min(demand / avail, 2.0) / 2.0
        else:
            tightness = 1.0 if demand > 0 else 0.0
            
        G.add_node(f'capacity_{good}', type=1, weight=tightness)
    
    # Stability constraint nodes (type 1) - for critical preference conflicts
    stability_constraints = 0
    for person1 in range(m):
        for person2 in range(person1 + 1, m):
            # Check if they have overlapping preferences that could cause instability
            p1_start = get_pref_start(person1)
            p2_start = get_pref_start(person2)
            
            # Get goods both people want
            p1_goods = set()
            p2_goods = set()
            
            for i in range(npref[person1]):
                pref_idx = p1_start + i
                if pref_idx < len(good_pref):
                    p1_goods.add(good_pref[pref_idx])
            
            for i in range(npref[person2]):
                pref_idx = p2_start + i
                if pref_idx < len(good_pref):
                    p2_goods.add(good_pref[pref_idx])
            
            # If they want the same scarce goods, create stability constraint
            common_goods = p1_goods.intersection(p2_goods)
            if common_goods:
                conflict_weight = 0
                for good_id in common_goods:
                    good_idx = good_id - 1  # Convert to 0-indexed
                    if 0 <= good_idx < n:
                        avail = available[good_idx] if good_idx < len(available) else 1
                        demand = good_demand[good_idx]
                        if demand > avail:  # Scarce good
                            conflict_weight += (demand - avail) / max(avail, 1)
                
                if conflict_weight > 0.5:  # Only create constraint for significant conflicts
                    constraint_weight = min(conflict_weight / 5.0, 1.0)
                    G.add_node(f'stability_{person1}_{person2}', type=1, weight=constraint_weight)
                    stability_constraints += 1
                    
                    # Connect to both people
                    G.add_edge(f'person_{person1}', f'stability_{person1}_{person2}', weight=0.8)
                    G.add_edge(f'person_{person2}', f'stability_{person1}_{person2}', weight=0.8)
    
    # Preference edges: people to goods they want
    for person in range(m):
        pref_start = get_pref_start(person)
        for i in range(npref[person]):
            pref_idx = pref_start + i
            if pref_idx < len(good_pref) and pref_idx < len(req_pref):
                good_id = good_pref[pref_idx] - 1  # Convert to 0-indexed
                req_amount = req_pref[pref_idx]
                
                if 0 <= good_id < n:
                    # Weight by preference rank (earlier preferences are stronger)
                    rank_weight = 1.0 - (i / max(npref[person], 1))
                    
                    # Weight by requirement amount relative to availability
                    avail = available[good_id] if good_id < len(available) else 1
                    req_weight = min(req_amount / max(avail, 1), 2.0) / 2.0
                    
                    # Combined edge weight
                    edge_weight = (rank_weight * 0.6 + req_weight * 0.4)
                    G.add_edge(f'person_{person}', f'good_{good_id}', weight=edge_weight)
    
    # Capacity constraint edges: goods to their capacity constraints
    for good in range(n):
        G.add_edge(f'good_{good}', f'capacity_{good}', weight=1.0)
    
    # Connect people to capacity constraints for goods they demand
    for person in range(m):
        pref_start = get_pref_start(person)
        for i in range(npref[person]):
            pref_idx = pref_start + i
            if pref_idx < len(good_pref):
                good_id = good_pref[pref_idx] - 1  # Convert to 0-indexed
                if 0 <= good_id < n:
                    # Weight by how much they contribute to capacity pressure
                    req_amount = req_pref[pref_idx] if pref_idx < len(req_pref) else 0
                    avail = available[good_id] if good_id < len(available) else 1
                    pressure = min(req_amount / max(avail, 1), 1.0)
                    
                    G.add_edge(f'person_{person}', f'capacity_{good_id}', weight=pressure)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()