#!/usr/bin/env python3
"""
Graph converter for stack-cutstock-cumu problem.
Created using subagent_prompt.md version: v_02

This problem is about cutting products from stock pieces with stack limits.
Key challenges: minimizing waste while respecting capacity and cumulative stack constraints.
The cumulative constraint limits how many stacks can be open simultaneously.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the stack-cutstock-cumu problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with products, cuts, and constraints
    - Products (type 0): Items to be cut, weighted by demand density
    - Cuts (type 2): Available cutting patterns, weighted by efficiency
    - Constraints (type 1): Capacity, demand, and stack limit constraints
    - Stack interaction edges reflect cumulative constraint complexity
    """
    # Extract problem data
    n = json_data.get('n', 0)  # number of products
    sl = json_data.get('sl', 1)  # stack limit
    s = json_data.get('s', 1)  # stock piece size
    k = json_data.get('k', 1)  # maximum cuts
    size = json_data.get('size', [])
    number = json_data.get('number', [])
    
    G = nx.Graph()
    
    if n == 0 or not size or not number:
        return G
    
    # Calculate derived metrics
    total_volume = sum(size[i] * number[i] for i in range(n))
    max_size = max(size) if size else 1
    max_number = max(number) if number else 1
    
    # Product nodes (type 0) - weighted by demand density and size difficulty
    for p in range(n):
        product_size = size[p] if p < len(size) else 1
        product_demand = number[p] if p < len(number) else 1
        
        # Demand density: how much of total demand this product represents
        demand_ratio = product_demand / max_number
        
        # Size difficulty: larger pieces are harder to fit efficiently
        size_difficulty = product_size / max_size
        
        # Combined weight with non-linear scaling
        weight = 0.5 * demand_ratio + 0.5 * math.sqrt(size_difficulty)
        
        G.add_node(f'product_{p}', type=0, weight=min(weight, 1.0))
    
    # Cut nodes (type 2) - represent cutting pattern resources
    for c in range(k):
        # Weight cuts by their potential efficiency (inverse of position)
        # Earlier cuts are more likely to be used efficiently
        efficiency = 1.0 - (c / max(k, 1))
        # Use exponential decay to emphasize preference for early cuts
        weight = math.exp(-2.0 * c / max(k, 1))
        
        G.add_node(f'cut_{c}', type=2, weight=weight)
    
    # Constraint nodes (type 1)
    
    # 1. Demand constraints - one per product
    for p in range(n):
        product_demand = number[p] if p < len(number) else 1
        # Weight by relative demand - high demand products create tighter constraints
        demand_tightness = product_demand / max_number
        G.add_node(f'demand_constraint_{p}', type=1, weight=demand_tightness)
    
    # 2. Capacity constraints - one per cut
    for c in range(k):
        # All capacity constraints are equally tight initially
        G.add_node(f'capacity_constraint_{c}', type=1, weight=0.8)
    
    # 3. Stack limit constraint (cumulative) - global constraint
    # Weight by how restrictive the stack limit is relative to number of products
    stack_pressure = min(1.0, n / max(sl, 1))
    stack_weight = 0.5 + 0.5 * stack_pressure  # Higher pressure = higher weight
    G.add_node('stack_limit_constraint', type=1, weight=stack_weight)
    
    # Edges: Product-Constraint relationships
    
    # Products to their demand constraints
    for p in range(n):
        G.add_edge(f'product_{p}', f'demand_constraint_{p}', weight=1.0)
    
    # Products to capacity constraints (through potential cut assignments)
    for p in range(n):
        product_size = size[p] if p < len(size) else 1
        for c in range(k):
            # Weight by how much of the stock this product would consume
            consumption_ratio = product_size / s
            weight = min(consumption_ratio * 1.5, 1.0)  # Scale up for visibility
            G.add_edge(f'product_{p}', f'capacity_constraint_{c}', weight=weight)
    
    # Products to stack limit constraint
    for p in range(n):
        product_demand = number[p] if p < len(number) else 1
        # Products with high demand contribute more to stack pressure
        stack_contribution = math.sqrt(product_demand / max_number)
        G.add_edge(f'product_{p}', 'stack_limit_constraint', weight=stack_contribution)
    
    # Cut-Constraint relationships
    
    # Cuts to their capacity constraints
    for c in range(k):
        G.add_edge(f'cut_{c}', f'capacity_constraint_{c}', weight=1.0)
    
    # Cuts to stack limit constraint (all cuts contribute to stack usage)
    for c in range(k):
        # Earlier cuts more likely to be used, so higher contribution to stack pressure
        usage_probability = 1.0 - (c / max(k, 1))
        weight = 0.3 + 0.7 * usage_probability
        G.add_edge(f'cut_{c}', 'stack_limit_constraint', weight=weight)
    
    # Product interaction edges for stack overlap potential
    # Products that might be cut together create stack dependencies
    for p1 in range(n):
        for p2 in range(p1 + 1, n):
            size1 = size[p1] if p1 < len(size) else 1
            size2 = size[p2] if p2 < len(size) else 1
            
            # If both products can fit on same stock piece, they might overlap in time
            if size1 + size2 <= s:
                # Weight by potential for temporal overlap in cutting
                overlap_potential = 1.0 - abs(size1 - size2) / max_size
                if overlap_potential > 0.5:  # Only add edge if significant overlap potential
                    weight = 0.3 * overlap_potential
                    G.add_edge(f'product_{p1}', f'product_{p2}', weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()