#!/usr/bin/env python3
"""
Graph converter for steelmillslab problem.
Created using subagent_prompt.md version: v_02

This problem is about steel mill slab optimization where orders with different sizes 
and colors must be assigned to slabs while minimizing waste.
Key challenges: Balancing bin packing efficiency with color compatibility constraints.
Each slab can hold at most 2 different colors, creating a complex optimization problem.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the steelmillslab problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with explicit constraint modeling
    - Order nodes (type 0): Decision variables representing orders to be assigned
    - Constraint nodes (type 1): Color and capacity constraints for slabs
    - Slab nodes (type 2): Resources representing available slabs
    - Model the complex interaction between bin packing and color constraints
    """
    # Access data directly from json_data dict
    nb_orders = json_data.get('nbOrders', 0)
    nb_colours = json_data.get('nbColours', 0) 
    ord_size = json_data.get('ordSize', [])
    ord_col = json_data.get('ordCol', [])
    sizes = json_data.get('sizes', [])
    
    # Create graph
    G = nx.Graph()
    
    if nb_orders == 0 or not ord_size or not ord_col:
        return G
    
    # Calculate some metrics for weight computation
    max_size = max(ord_size) if ord_size else 1
    max_slab_size = max(sizes) if sizes else max_size
    
    # Add order nodes (type 0) - the decision variables
    for i in range(nb_orders):
        order_size = ord_size[i] if i < len(ord_size) else 0
        order_color = ord_col[i] if i < len(ord_col) else 1
        
        # Weight by relative size and color rarity
        color_count = sum(1 for c in ord_col if c == order_color)
        color_rarity = 1.0 - (color_count / nb_orders)
        size_importance = order_size / max_size
        
        # Combine size importance with color rarity using non-linear weighting
        weight = 0.3 * size_importance + 0.7 * math.sqrt(color_rarity)
        
        G.add_node(f'order_{i}', type=0, weight=min(weight, 1.0))
    
    # Add slab nodes (type 2) - the resources
    nb_slabs = nb_orders  # As per model: nbSlabs = nbOrders
    for s in range(nb_slabs):
        # Weight slabs by their potential utilization difficulty
        # Slabs that will be harder to fill efficiently get higher weight
        slab_weight = 0.5 + 0.5 * math.exp(-s / (nb_slabs * 0.3))
        G.add_node(f'slab_{s}', type=2, weight=slab_weight)
    
    # Add color constraint nodes (type 1) - each slab can have at most 2 colors
    for s in range(nb_slabs):
        # Weight by how restrictive this constraint is likely to be
        # More colors in the problem make this constraint tighter
        color_tightness = min(nb_colours / 10.0, 1.0)  # Normalize color pressure
        G.add_node(f'color_constraint_{s}', type=1, weight=color_tightness)
    
    # Add capacity constraint nodes (type 1) - bin packing constraints for each size
    for size_idx, size_capacity in enumerate(sizes):
        # Calculate total demand that could use this size
        demand_for_size = sum(ord_size[i] for i in range(nb_orders) 
                            if ord_size[i] <= size_capacity)
        
        # Tightness based on demand vs available capacity
        if size_capacity > 0:
            demand_ratio = demand_for_size / (size_capacity * nb_slabs)
            tightness = min(demand_ratio, 1.0)
        else:
            tightness = 0.1  # Low weight for size 0
            
        G.add_node(f'capacity_{size_idx}', type=1, weight=tightness)
    
    # Add bipartite edges: orders to slabs (potential assignments)
    for i in range(nb_orders):
        order_size = ord_size[i] if i < len(ord_size) else 0
        for s in range(nb_slabs):
            # Edge weight based on how well this order fits potential slab capacities
            best_fit_score = 0.0
            for size_capacity in sizes:
                if size_capacity >= order_size:
                    # Better fit = higher weight (less waste)
                    fit_efficiency = order_size / size_capacity
                    best_fit_score = max(best_fit_score, fit_efficiency)
            
            if best_fit_score > 0:
                G.add_edge(f'order_{i}', f'slab_{s}', weight=best_fit_score)
    
    # Add edges: orders to color constraints
    for i in range(nb_orders):
        for s in range(nb_slabs):
            # Each order participates in the color constraint for each slab
            G.add_edge(f'order_{i}', f'color_constraint_{s}', weight=0.8)
    
    # Add edges: orders to capacity constraints
    for i in range(nb_orders):
        order_size = ord_size[i] if i < len(ord_size) else 0
        for size_idx, size_capacity in enumerate(sizes):
            if order_size <= size_capacity:
                # Weight by how much of the capacity this order uses
                utilization = order_size / size_capacity if size_capacity > 0 else 0
                G.add_edge(f'order_{i}', f'capacity_{size_idx}', weight=utilization)
    
    # Add conflict edges between orders with same color that are large
    # This models the difficulty of placing multiple large orders of same color
    for i in range(nb_orders):
        for j in range(i + 1, nb_orders):
            if (i < len(ord_col) and j < len(ord_col) and 
                ord_col[i] == ord_col[j] and 
                ord_size[i] > max_size * 0.7 and ord_size[j] > max_size * 0.7):
                
                # Conflict strength based on combined size pressure
                conflict_weight = (ord_size[i] + ord_size[j]) / (2 * max_size)
                G.add_edge(f'order_{i}', f'order_{j}', weight=min(conflict_weight, 1.0))
    
    # Add edges: slabs to their constraints
    for s in range(nb_slabs):
        G.add_edge(f'slab_{s}', f'color_constraint_{s}', weight=1.0)
        
        # Connect slabs to relevant capacity constraints
        for size_idx in range(len(sizes)):
            G.add_edge(f'slab_{s}', f'capacity_{size_idx}', weight=0.6)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()