#!/usr/bin/env python3
"""
Graph converter for steiner-triples problem.
Created using subagent_prompt.md version: v_02

This problem is about finding Steiner triplets - sets of 3 numbers from 1 to n 
such that any two triplets have at most one element in common.
Key challenges: Intersection constraints between sets, exponential search space,
symmetry breaking requirements.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Steiner triples problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with explicit constraint nodes
    - Element nodes (type 0): Numbers 1 to n that form triplets
    - Triplet nodes (type 1): The nb triplet decision variables 
    - Intersection constraint nodes (type 1): Pairwise intersection constraints
    - Element-triplet membership edges show participation
    - Triplet-constraint edges show which constraints apply to which triplets
    
    Difficulty factors:
    - Number of elements (n) affects complexity quadratically
    - Number of triplets nb = n*(n-1)/6 grows quadratically
    - Number of intersection constraints grows as nb*(nb-1)/2
    """
    # Access data directly from json_data dict
    n = json_data.get('n', 0)
    
    if n < 3:
        # Degenerate case
        G = nx.Graph()
        G.add_node('dummy', type=0, weight=1.0)
        return G
    
    # Calculate number of triplets
    nb = n * (n - 1) // 6
    
    # Create graph
    G = nx.Graph()
    
    # Element nodes (type 0) - the numbers 1 to n
    # Weight by centrality - elements in middle range are most constrained
    for elem in range(1, n + 1):
        # Central elements are more constrained (used in more triplets)
        centrality = 1.0 - abs(elem - (n + 1) / 2) / (n / 2) if n > 1 else 1.0
        # Add some non-linearity
        weight = math.pow(centrality, 1.5) * 0.8 + 0.2  # Range [0.2, 1.0]
        G.add_node(f'elem_{elem}', type=0, weight=weight)
    
    # Triplet decision variable nodes (type 1)
    # Each triplet has weight based on its potential constraint burden
    for t in range(nb):
        # Later triplets in ordering are more constrained by symmetry breaking
        ordering_pressure = (t + 1) / nb
        # Add exponential scaling for constraint pressure
        weight = 0.3 + 0.7 * math.pow(ordering_pressure, 0.8)
        G.add_node(f'triplet_{t}', type=1, weight=weight)
    
    # Intersection constraint nodes (type 1)
    # Each pair of triplets must have at most 1 element in common
    constraint_id = 0
    for i in range(nb):
        for j in range(i + 1, nb):
            # Weight by position - later constraints are harder due to accumulated restrictions
            position_factor = constraint_id / (nb * (nb - 1) / 2) if nb > 1 else 0
            # Use sqrt for moderate non-linearity
            weight = 0.5 + 0.5 * math.sqrt(position_factor)
            G.add_node(f'intersect_{i}_{j}', type=1, weight=weight)
            constraint_id += 1
    
    # Cardinality constraint nodes (type 1) - each triplet must have exactly 3 elements
    for t in range(nb):
        # All cardinality constraints are equally important
        G.add_node(f'card_{t}', type=1, weight=0.9)
    
    # Element-triplet membership edges
    # Each element can potentially be in multiple triplets
    # Weight by scarcity - with nb triplets and each having 3 elements,
    # each element appears in approximately 3*nb/n triplets
    expected_membership = (3 * nb) / n if n > 0 else 1
    membership_scarcity = min(expected_membership / n, 1.0)
    
    for elem in range(1, n + 1):
        for t in range(nb):
            # All potential memberships have same weight (actual membership determined by solver)
            G.add_edge(f'elem_{elem}', f'triplet_{t}', weight=membership_scarcity)
    
    # Triplet-intersection constraint edges
    # Connect each triplet to all intersection constraints it participates in
    for i in range(nb):
        for j in range(i + 1, nb):
            # Both triplets participate in this intersection constraint
            # Weight by constraint burden - more constraints per triplet = higher weight
            constraint_density = (nb - 1) / nb if nb > 1 else 1.0
            G.add_edge(f'triplet_{i}', f'intersect_{i}_{j}', weight=constraint_density)
            G.add_edge(f'triplet_{j}', f'intersect_{i}_{j}', weight=constraint_density)
    
    # Triplet-cardinality constraint edges
    # Each triplet is connected to its cardinality constraint
    for t in range(nb):
        G.add_edge(f'triplet_{t}', f'card_{t}', weight=1.0)
    
    # Add symmetry breaking constraint node (type 1)
    # The ordering constraint affects all adjacent triplet pairs
    if nb > 1:
        G.add_node('symmetry_breaking', type=1, weight=0.7)
        for t in range(nb - 1):
            # Weight by position - later comparisons are more constrained
            position_weight = 0.6 + 0.4 * (t / (nb - 1))
            G.add_edge(f'triplet_{t}', 'symmetry_breaking', weight=position_weight)
            G.add_edge(f'triplet_{t+1}', 'symmetry_breaking', weight=position_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()