#!/usr/bin/env python3
"""
Graph converter for still-life problem.
Created using subagent_prompt.md version: v_02

This problem is about finding stable patterns in Conway's Game of Life (still-lifes).
Key challenges: complex neighborhood constraints, balancing living cells with stability requirements,
and wastage optimization. The problem involves finding maximal configurations where each cell
follows Game of Life rules to remain stable.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the still-life problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the cellular automaton as a bipartite graph where:
    - Cell nodes (type 0) represent decision variables for each board position
    - Constraint nodes (type 1) represent different types of constraints:
      * Still-life constraints for each 3x3 neighborhood
      * Boundary constraints
      * Wastage constraints
    - Edge weights reflect constraint tightness and cell importance
    
    Key insight: Still-life difficulty comes from balancing local neighborhood
    rules with global optimization, so we model both local and global structure.
    """
    n = json_data.get('n', 9)
    
    G = nx.Graph()
    
    # === CELL NODES (Type 0) ===
    # Weight cells by their strategic importance:
    # - Corner cells have fewer neighbors (easier to satisfy)
    # - Center cells have more constraints (harder)
    # - Edge cells are intermediate
    
    for i in range(1, n+1):
        for j in range(1, n+1):
            # Calculate position-based difficulty
            # Distance from center influences constraint complexity
            center_i, center_j = (n+1) // 2, (n+1) // 2
            dist_from_center = abs(i - center_i) + abs(j - center_j)
            max_dist = abs(1 - center_i) + abs(1 - center_j)
            
            # Corner and edge effects - cells near boundaries have fewer neighbors
            edge_distance = min(i-1, j-1, n-i, n-j)
            
            # Central cells are more constrained (higher weight)
            # But use non-linear scaling to emphasize differences
            centrality = 1.0 - (dist_from_center / max_dist) if max_dist > 0 else 0.5
            boundary_effect = math.exp(-2.0 * edge_distance / n)  # Exponential decay from edges
            
            # Combine factors: central cells with boundary effects get higher weights
            cell_weight = 0.3 + 0.4 * centrality + 0.3 * boundary_effect
            cell_weight = min(cell_weight, 1.0)
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=cell_weight)
    
    # === CONSTRAINT NODES (Type 1) ===
    
    # 1. Still-life constraints - one for each 3x3 neighborhood
    # These are the most complex constraints with 9 variables each
    for i in range(1, n+1):
        for j in range(1, n+1):
            # Weight by complexity of neighborhood constraint
            # Still-life rules involve complex logical combinations
            # More central neighborhoods have more "room" for complex patterns
            center_i, center_j = (n+1) // 2, (n+1) // 2
            dist_from_center = abs(i - center_i) + abs(j - center_j)
            max_dist = abs(1 - center_i) + abs(1 - center_j)
            
            # Central still-life constraints are more critical (higher weight)
            constraint_weight = 0.7 + 0.3 * (1.0 - dist_from_center / max_dist) if max_dist > 0 else 0.85
            
            G.add_node(f'stilllife_{i}_{j}', type=1, weight=constraint_weight)
    
    # 2. Boundary constraints - cells on edges must satisfy special rules
    # These have fewer variables but are critical for stability
    boundary_positions = []
    
    # Top and bottom edge constraints
    for i in [1, n]:
        for j in range(2, n):  # Middle positions on edges
            boundary_positions.append(f'boundary_edge_{i}_{j}')
            G.add_node(f'boundary_edge_{i}_{j}', type=1, weight=0.8)
    
    # Left and right edge constraints  
    for j in [1, n]:
        for i in range(2, n):  # Middle positions on edges
            boundary_positions.append(f'boundary_edge_{i}_{j}')
            G.add_node(f'boundary_edge_{i}_{j}', type=1, weight=0.8)
    
    # 3. Wastage constraints - relate boundary wastage to internal cells
    # These help optimize the objective function
    for i in range(1, n+1):
        # Boundary wastage constraints are medium complexity
        G.add_node(f'wastage_boundary_{i}', type=1, weight=0.6)
    
    # 4. Global wastage sum constraint (most complex)
    # This constraint involves all wastage variables
    total_variables = (n+2) * (n+2)  # All w variables
    complexity = math.log(total_variables) / math.log(n*n)  # Logarithmic scaling
    G.add_node('global_wastage', type=1, weight=min(0.9, 0.5 + 0.1 * complexity))
    
    # === EDGES ===
    
    # 1. Cell participation in still-life constraints
    # Each cell participates in up to 9 still-life constraints (as center or neighbor)
    for i in range(1, n+1):
        for j in range(1, n+1):
            cell_node = f'cell_{i}_{j}'
            
            # Connect to still-life constraints where this cell appears
            for ni in range(max(1, i-1), min(n+1, i+2)):
                for nj in range(max(1, j-1), min(n+1, j+2)):
                    constraint_node = f'stilllife_{ni}_{nj}'
                    
                    # Weight by role in constraint:
                    # Center cell (the one being evaluated) has highest weight
                    # Neighbors have weights based on their influence
                    if ni == i and nj == j:
                        # This cell is the center of the still-life constraint
                        participation_weight = 1.0
                    else:
                        # This cell is a neighbor - weight by distance and diagonal effect
                        dist = abs(ni - i) + abs(nj - j)
                        is_diagonal = (abs(ni - i) == 1 and abs(nj - j) == 1)
                        
                        if is_diagonal:
                            participation_weight = 0.6  # Diagonal neighbors are important
                        else:
                            participation_weight = 0.8  # Orthogonal neighbors are very important
                    
                    G.add_edge(cell_node, constraint_node, weight=participation_weight)
    
    # 2. Cell participation in boundary constraints
    # Edge cells participate in boundary constraints
    for i in [1, n]:
        for j in range(2, n):
            # Top/bottom edge cells
            for offset in [-1, 0, 1]:
                if 1 <= j + offset <= n:
                    cell_node = f'cell_{i}_{j + offset}'
                    constraint_node = f'boundary_edge_{i}_{j}'
                    weight = 0.9 if offset == 0 else 0.7  # Center cell more important
                    G.add_edge(cell_node, constraint_node, weight=weight)
    
    for j in [1, n]:
        for i in range(2, n):
            # Left/right edge cells
            for offset in [-1, 0, 1]:
                if 1 <= i + offset <= n:
                    cell_node = f'cell_{i + offset}_{j}'
                    constraint_node = f'boundary_edge_{i}_{j}'
                    weight = 0.9 if offset == 0 else 0.7  # Center cell more important
                    G.add_edge(cell_node, constraint_node, weight=weight)
    
    # 3. Boundary cells participate in wastage constraints
    # Each edge position affects wastage calculation
    for i in range(1, n+1):
        # Top and bottom edges
        for j in [1, n]:
            if j <= n:
                cell_node = f'cell_{i}_{j}'
                wastage_node = f'wastage_boundary_{i}'
                G.add_edge(cell_node, wastage_node, weight=0.7)
        
        # Left and right edges  
        for j in [1, n]:
            if i <= n:
                cell_node = f'cell_{j}_{i}'  # Note: swapped for left/right
                wastage_node = f'wastage_boundary_{i}'
                if cell_node in G.nodes:  # Avoid duplicate edges
                    G.add_edge(cell_node, wastage_node, weight=0.7)
    
    # 4. Global wastage connections
    # Connect global wastage to some key cells and local wastage constraints
    center_i, center_j = (n+1) // 2, (n+1) // 2
    
    # Connect to central cells (they most influence the objective)
    for i in range(max(1, center_i-1), min(n+1, center_i+2)):
        for j in range(max(1, center_j-1), min(n+1, center_j+2)):
            cell_node = f'cell_{i}_{j}'
            G.add_edge(cell_node, 'global_wastage', weight=0.5)
    
    # Connect to wastage boundary constraints
    for i in range(1, n+1):
        wastage_node = f'wastage_boundary_{i}'
        G.add_edge(wastage_node, 'global_wastage', weight=0.8)
    
    # 5. Add some conflict edges between cells that would create unstable patterns
    # High-density regions where too many cells are alive create instability
    for i in range(1, n):
        for j in range(1, n):
            # 2x2 squares of cells would violate still-life properties if all alive
            # Add conflict edges within 2x2 regions with exponential decay weight
            cell1 = f'cell_{i}_{j}'
            cell2 = f'cell_{i+1}_{j}'
            cell3 = f'cell_{i}_{j+1}'
            cell4 = f'cell_{i+1}_{j+1}'
            
            # Add pairwise conflicts within 2x2 regions
            # Weight decreases with problem size (larger boards have more flexibility)
            conflict_weight = 0.3 * math.exp(-n / 10.0)
            
            pairs = [(cell1, cell2), (cell1, cell3), (cell1, cell4), 
                    (cell2, cell3), (cell2, cell4), (cell3, cell4)]
            
            for c1, c2 in pairs:
                if not G.has_edge(c1, c2):  # Avoid duplicate edges
                    G.add_edge(c1, c2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()