#!/usr/bin/env python3
"""
Graph converter for still_life problem.
Created using subagent_prompt.md version: v_02

This problem is about finding still life configurations in Conway's Game of Life.
Key challenges: Complex neighbor interactions, boundary constraints, maximizing live cells while maintaining stability.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the still life problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with cells and constraints
    - Cell nodes (type 0): represent grid positions with location-based weights
    - Constraint nodes (type 1): neighbor count constraints, boundary constraints
    - Edges connect cells to constraints they participate in
    - Weights reflect constraint tightness and cell importance
    """
    n = json_data.get('n', 5)
    
    G = nx.Graph()
    
    # Cell nodes (type 0) - decision variables for each grid position
    for r in range(1, n+1):
        for c in range(1, n+1):
            # Weight based on position criticality
            # Edge and corner cells are more constrained due to boundary conditions
            # Center cells have more freedom but also more neighbor interactions
            
            # Distance from edge (0 = on edge, higher = more central)
            edge_dist = min(r-1, c-1, n-r, n-c)
            # Normalize edge distance
            max_edge_dist = (n-1) // 2
            edge_factor = edge_dist / max(max_edge_dist, 1) if max_edge_dist > 0 else 0
            
            # Corner penalty (corners are most constrained)
            is_corner = (r == 1 or r == n) and (c == 1 or c == n)
            corner_penalty = 0.3 if is_corner else 0
            
            # Central positions are critical for pattern formation
            center_r, center_c = (n+1)/2, (n+1)/2
            center_dist = math.sqrt((r - center_r)**2 + (c - center_c)**2)
            max_center_dist = math.sqrt(2) * (n-1)/2
            centrality = 1.0 - (center_dist / max(max_center_dist, 1))
            
            # Combine factors - use non-linear weighting
            weight = (0.4 * centrality + 0.4 * edge_factor + 0.2) - corner_penalty
            weight = max(0.1, min(1.0, weight))
            
            G.add_node(f'cell_{r}_{c}', type=0, weight=weight)
    
    # Constraint nodes (type 1) - neighbor count constraints for each cell
    for r in range(1, n+1):
        for c in range(1, n+1):
            # Each cell has a neighbor count constraint
            # Weight by the number of potential neighbors (scope)
            neighbor_count = 0
            for rr in range(max(1, r-1), min(n, r+1)+1):
                for cc in range(max(1, c-1), min(n, c+1)+1):
                    if rr != r or cc != c:
                        neighbor_count += 1
            
            # Normalize by maximum possible neighbors (8)
            scope_weight = neighbor_count / 8.0
            
            # Edge cells have tighter constraints due to fewer neighbors
            is_edge = r == 1 or r == n or c == 1 or c == n
            tightness = 0.8 if is_edge else 0.6
            
            weight = scope_weight * tightness
            G.add_node(f'neighbor_constraint_{r}_{c}', type=1, weight=weight)
    
    # Still life rule constraints (type 1) - separate from neighbor counting
    for r in range(1, n+1):
        for c in range(1, n+1):
            # Dead cell rule: can't have exactly 3 neighbors if dead
            G.add_node(f'dead_rule_{r}_{c}', type=1, weight=0.7)
            # Live cell rule: must have 2-3 neighbors if alive
            G.add_node(f'live_rule_{r}_{c}', type=1, weight=0.8)
    
    # Boundary constraints (type 1) - prevent edge activation patterns
    boundary_constraints = 0
    for i in range(2, n):  # Only interior positions along edges
        # Top and bottom boundary constraints
        G.add_node(f'boundary_top_{i}', type=1, weight=0.9)
        G.add_node(f'boundary_bottom_{i}', type=1, weight=0.9)
        # Left and right boundary constraints  
        G.add_node(f'boundary_left_{i}', type=1, weight=0.9)
        G.add_node(f'boundary_right_{i}', type=1, weight=0.9)
        boundary_constraints += 4
    
    # Edges: Cell participation in constraints
    
    # 1. Neighbor count constraint edges
    for r in range(1, n+1):
        for c in range(1, n+1):
            cell = f'cell_{r}_{c}'
            constraint = f'neighbor_constraint_{r}_{c}'
            
            # Each cell participates in its own neighbor constraint
            G.add_edge(cell, constraint, weight=1.0)
            
            # Each cell also contributes to its neighbors' constraints
            for rr in range(max(1, r-1), min(n, r+1)+1):
                for cc in range(max(1, c-1), min(n, c+1)+1):
                    if rr != r or cc != c:  # Not the cell itself
                        neighbor_constraint = f'neighbor_constraint_{rr}_{cc}'
                        if neighbor_constraint in [node for node in G.nodes()]:
                            # Weight by distance (closer neighbors have stronger influence)
                            dist = math.sqrt((r-rr)**2 + (c-cc)**2)
                            weight = math.exp(-dist)  # Exponential decay
                            G.add_edge(cell, neighbor_constraint, weight=weight)
    
    # 2. Still life rule constraint edges
    for r in range(1, n+1):
        for c in range(1, n+1):
            cell = f'cell_{r}_{c}'
            
            # Connect to both dead and live rule constraints
            G.add_edge(cell, f'dead_rule_{r}_{c}', weight=0.8)
            G.add_edge(cell, f'live_rule_{r}_{c}', weight=0.9)
            
            # Also connect neighbors to this cell's rules (they influence the decision)
            for rr in range(max(1, r-1), min(n, r+1)+1):
                for cc in range(max(1, c-1), min(n, c+1)+1):
                    if rr != r or cc != c:
                        neighbor_cell = f'cell_{rr}_{cc}'
                        if neighbor_cell in [node for node in G.nodes()]:
                            # Neighbors influence the still life rules
                            G.add_edge(neighbor_cell, f'dead_rule_{r}_{c}', weight=0.5)
                            G.add_edge(neighbor_cell, f'live_rule_{r}_{c}', weight=0.6)
    
    # 3. Boundary constraint edges
    for i in range(2, n):
        # Top boundary: sum of cells (i-1,1), (i,1), (i+1,1) < 3
        for j in range(i-1, i+2):
            if 1 <= j <= n:
                G.add_edge(f'cell_{j}_1', f'boundary_top_{i}', weight=0.8)
        
        # Bottom boundary: sum of cells (i-1,n), (i,n), (i+1,n) < 3  
        for j in range(i-1, i+2):
            if 1 <= j <= n:
                G.add_edge(f'cell_{j}_{n}', f'boundary_bottom_{i}', weight=0.8)
        
        # Left boundary: sum of cells (1,i-1), (1,i), (1,i+1) < 3
        for j in range(i-1, i+2):
            if 1 <= j <= n:
                G.add_edge(f'cell_1_{j}', f'boundary_left_{i}', weight=0.8)
        
        # Right boundary: sum of cells (n,i-1), (n,i), (n,i+1) < 3
        for j in range(i-1, i+2):
            if 1 <= j <= n:
                G.add_edge(f'cell_{n}_{j}', f'boundary_right_{i}', weight=0.8)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()