#!/usr/bin/env python3
"""
Graph converter for Still Life problem.
Created using subagent_prompt.md version: v_02

This problem is about finding maximum still life patterns in Conway's Game of Life.
Key challenges: cells must satisfy neighbor count constraints for stability, 
maximizing alive cells while maintaining equilibrium state.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the still life problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with cells and stability constraints
    - Cell nodes (type 0): decision variables for alive/dead state
    - Stability constraint nodes (type 1): Game of Life rules for each cell
    - Border constraint nodes (type 1): enforce border cells are dead
    - Edge weights reflect constraint tightness and cell importance
    """
    n = json_data.get('n', 3)  # rows
    m = json_data.get('m', 3)  # columns
    
    G = nx.Graph()
    
    # Cell nodes (type 0) - decision variables for alive/dead state
    # Weight based on distance from border (central cells more important/constrained)
    for i in range(1, n+1):  # only internal cells (1 to n, 1 to m)
        for j in range(1, m+1):
            # Central cells are more constrained and valuable
            dist_from_edge = min(i, j, n+1-i, m+1-j)
            max_dist = min(n//2, m//2) + 1
            centrality = dist_from_edge / max_dist if max_dist > 0 else 0.5
            
            # Higher weight for more central cells (more constrained)
            weight = 0.3 + 0.7 * centrality
            G.add_node(f'cell_{i}_{j}', type=0, weight=weight)
    
    # Stability constraint nodes (type 1) - Game of Life rules
    # Each internal cell has a stability constraint
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Count how many neighbors this cell has (cells on edge have fewer neighbors)
            neighbor_count = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_count += 1
            
            # Cells with more neighbors have tighter constraints
            # Maximum neighbors is 8, minimum is 3 (corner cells)
            constraint_tightness = neighbor_count / 8.0
            G.add_node(f'stability_{i}_{j}', type=1, weight=constraint_tightness)
    
    # Border constraint nodes (type 1) - enforce dead border
    # These are very tight constraints (weight close to 1.0)
    border_cells = []
    
    # Top and bottom borders
    for j in range(-1, m+3):
        for i in [-1, 0, n+1, n+2]:
            border_cells.append((i, j))
    
    # Left and right borders  
    for i in range(-1, n+3):
        for j in [-1, 0, m+1, m+2]:
            border_cells.append((i, j))
    
    # Create constraint nodes for border enforcement
    G.add_node('border_constraint', type=1, weight=1.0)
    
    # Bipartite edges: cells to their stability constraints
    for i in range(1, n+1):
        for j in range(1, m+1):
            cell_node = f'cell_{i}_{j}'
            stability_node = f'stability_{i}_{j}'
            
            # Edge weight represents how critical this cell is for the constraint
            # Corner and edge cells have fewer neighbors, so violations are more impactful
            neighbor_count = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_count += 1
            
            # Cells with fewer neighbors are more constrained
            edge_weight = 1.0 - (neighbor_count - 3) / 5.0  # normalize to [0.4, 1.0]
            G.add_edge(cell_node, stability_node, weight=edge_weight)
    
    # Edges from cells to neighbor stability constraints
    # Each cell participates in multiple stability constraints (its neighbors')
    for i in range(1, n+1):
        for j in range(1, m+1):
            cell_node = f'cell_{i}_{j}'
            
            # Connect to all neighbor stability constraints
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_stability = f'stability_{ni}_{nj}'
                        
                        # Weight based on geometric distance (closer = stronger influence)
                        distance = math.sqrt(di*di + dj*dj)
                        influence = math.exp(-distance)  # exponential decay
                        G.add_edge(cell_node, neighbor_stability, weight=influence)
    
    # Connect border cells to border constraint (if they exist as decision variables)
    # In this problem, border cells are fixed to 0, but we model the constraint anyway
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Cells closer to border are more affected by border constraint
            dist_to_border = min(i, j, n+1-i, m+1-j)
            if dist_to_border <= 2:  # cells close to border
                border_influence = math.exp(-dist_to_border)
                G.add_edge(f'cell_{i}_{j}', 'border_constraint', weight=border_influence)
    
    # Add conflict edges between cells that would create unstable patterns
    # Cells that are too close together may create oscillating patterns
    for i in range(1, n+1):
        for j in range(1, m+1):
            for i2 in range(i, n+1):
                for j2 in range(j, m+1):
                    if i == i2 and j == j2:
                        continue
                    
                    distance = math.sqrt((i2-i)**2 + (j2-j)**2)
                    # Very close cells (distance 1-2) may conflict
                    if distance <= 2.0:
                        # Conflict strength decreases with distance
                        conflict_strength = math.exp(-distance)
                        # Only add significant conflicts
                        if conflict_strength > 0.3:
                            G.add_edge(f'cell_{i}_{j}', f'cell_{i2}_{j2}', 
                                     weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()