#!/usr/bin/env python3
"""
Graph converter for still_life_free problem.
Created using subagent_prompt.md version: v_02

This problem is about finding maximum still life configurations in Conway's Game of Life.
A still life is a configuration where all cells remain unchanged in the next generation.
Key challenges: balancing alive cells while satisfying Game of Life rules, complex neighbor interactions.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the still life problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph capturing Game of Life constraints
    - Cell nodes (type 0): Each cell position with centrality-based weights
    - Rule constraint nodes (type 1): Game of Life rules for each cell
    - Neighbor constraint nodes (type 1): Sum constraints for neighbor counting
    - Edges connect cells to constraints they participate in
    """
    n = json_data.get('n', 3)  # number of rows
    m = json_data.get('m', 3)  # number of columns
    
    G = nx.Graph()
    
    # Add cell nodes (type 0) - decision variables for alive/dead
    for i in range(1, n+1):  # 1..n as in MiniZinc
        for j in range(1, m+1):  # 1..m as in MiniZinc
            # Weight based on centrality - central cells are more constrained
            # and harder to satisfy due to more neighbors
            center_i, center_j = (n+1)/2, (m+1)/2
            dist_from_center = math.sqrt((i - center_i)**2 + (j - center_j)**2)
            max_dist = math.sqrt((center_i - 1)**2 + (center_j - 1)**2)
            
            # Central cells are more important (higher weight)
            centrality = 1.0 - (dist_from_center / max_dist) if max_dist > 0 else 1.0
            
            # Also consider edge effects - corner cells have fewer neighbors
            neighbor_count = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_count += 1
            
            # Cells with more neighbors are more constrained
            max_neighbors = 8
            neighbor_factor = neighbor_count / max_neighbors
            
            # Combine centrality and neighbor effects
            weight = 0.6 * centrality + 0.4 * neighbor_factor
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=weight)
    
    # Add constraint nodes for Game of Life rules (type 1)
    # Each cell has a rule constraint: alive iff 2-3 alive neighbors
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Weight constraint by how many neighbors it involves
            neighbor_positions = []
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_positions.append((ni, nj))
            
            # Constraints with more variables are generally harder
            scope_size = len(neighbor_positions) + 1  # +1 for the cell itself
            max_scope = 9  # maximum 8 neighbors + 1 cell
            scope_weight = scope_size / max_scope
            
            # Corner and edge constraints are typically easier
            is_corner = (i in [1, n]) and (j in [1, m])
            is_edge = (i in [1, n]) or (j in [1, m])
            
            if is_corner:
                difficulty = 0.3
            elif is_edge:
                difficulty = 0.6
            else:
                difficulty = 1.0  # Interior cells are hardest
                
            # Combine scope and position difficulty
            weight = 0.7 * scope_weight + 0.3 * difficulty
            
            G.add_node(f'rule_{i}_{j}', type=1, weight=weight)
    
    # Add neighbor sum constraint nodes (type 1)
    # Each cell has a sum constraint for counting neighbors
    for i in range(1, n+1):
        for j in range(1, m+1):
            neighbor_count = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_count += 1
            
            # Weight by number of neighbors involved
            weight = neighbor_count / 8.0
            G.add_node(f'sum_{i}_{j}', type=1, weight=weight)
    
    # Add optimization constraint node (type 1)
    # The cost maximization creates interdependencies
    total_cells = n * m
    # Larger grids have more complex optimization landscapes
    complexity = min(1.0, math.log(total_cells) / math.log(64))  # log scale
    G.add_node('cost_constraint', type=1, weight=complexity)
    
    # Add bipartite edges: cells to constraints they participate in
    for i in range(1, n+1):
        for j in range(1, m+1):
            cell = f'cell_{i}_{j}'
            
            # Connect to own rule constraint
            G.add_edge(cell, f'rule_{i}_{j}', weight=1.0)
            
            # Connect to own sum constraint  
            G.add_edge(cell, f'sum_{i}_{j}', weight=1.0)
            
            # Connect to cost constraint (all cells contribute)
            G.add_edge(cell, 'cost_constraint', weight=1.0)
            
            # Connect to neighbor constraints where this cell contributes
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        # This cell contributes to neighbor's sum constraint
                        # Weight by geometric distance (closer = stronger influence)
                        distance = math.sqrt(di*di + dj*dj)
                        weight = math.exp(-distance)  # Exponential decay
                        G.add_edge(cell, f'sum_{ni}_{nj}', weight=weight)
                        
                        # This cell affects neighbor's rule constraint
                        G.add_edge(cell, f'rule_{ni}_{nj}', weight=weight)
    
    # Add some direct conflict edges for cells that create impossible configurations
    # In still life problems, certain patterns are inherently unstable
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Look for potential oscillator patterns that would violate still life
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if abs(di) + abs(dj) == 1:  # Only direct neighbors
                        ni, nj = i + di, j + dj
                        if 1 <= ni <= n and 1 <= nj <= m:
                            # Adjacent cells in small grids create strong constraints
                            if n <= 4 and m <= 4:
                                # Small grids: adjacent cells have high interaction
                                G.add_edge(f'cell_{i}_{j}', f'cell_{ni}_{nj}', 
                                          weight=0.8)
                            else:
                                # Larger grids: weaker direct interactions
                                G.add_edge(f'cell_{i}_{j}', f'cell_{ni}_{nj}', 
                                          weight=0.3)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()