#!/usr/bin/env python3
"""
Graph converter for still_life_full_border problem.
Created using subagent_prompt.md version: v_02

This problem is about finding the maximum number of living cells in a Conway's Game of Life
configuration that remains stable (still life) with a full border of living cells.
Key challenges: Balancing the Game of Life stability rules with maximizing living cells,
especially with the constraint that all border cells must be alive.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the still life problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model this as a cellular automaton constraint problem
    - Cell nodes (type 0): Interior grid cells that can be alive/dead
    - Neighbor constraint nodes (type 1): Game of Life rules for each cell
    - Border constraint nodes (type 1): Enforce border cells are alive
    - Stability constraint node (type 1): Global still life constraint
    
    The graph captures the local neighborhood dependencies that make this problem challenging.
    Border cells create fixed constraints that propagate into the interior.
    """
    # Access data directly from json_data dict
    n = json_data.get('n', 3)  # rows
    m = json_data.get('m', 3)  # columns
    
    # Create graph
    G = nx.Graph()
    
    # Variable nodes: Interior cells (1..n, 1..m) that can be alive/dead
    # Border cells are fixed to 1, so not decision variables
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Corner cells are most constrained (fewer free neighbors)
            # Center cells have more freedom
            distance_from_center = abs(i - (n+1)/2) + abs(j - (m+1)/2)
            max_distance = (n-1)/2 + (m-1)/2
            
            # Corner and edge cells are more constrained due to fixed border
            if (i == 1 or i == n) and (j == 1 or j == m):
                # Corner cells - most constrained
                constraint_level = 1.0
            elif i == 1 or i == n or j == 1 or j == m:
                # Edge cells - highly constrained  
                constraint_level = 0.8
            else:
                # Interior cells - weight by distance from center (exponential decay)
                # Cells further from center are more constrained by border effects
                constraint_level = 0.3 + 0.4 * math.exp(-2.0 * distance_from_center / max_distance)
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=constraint_level)
    
    # Constraint nodes: Game of Life rules for each interior cell
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Each cell has its own neighborhood constraint
            # Weight by complexity of the constraint (number of neighbors)
            neighbor_count = 0
            
            # Count actual neighbors (including border effects)
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    # Border cells (ni=1,n or nj=1,m) are fixed to 1
                    # Boundary cells (ni=0,n+1 or nj=0,m+1) are fixed to 0
                    neighbor_count += 1
            
            # Constraint tightness based on how many neighbors are fixed
            # Border neighbors (fixed to 1) make constraint tighter
            border_neighbors = 0
            boundary_neighbors = 0
            
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if ni == 0 or ni == n+1 or nj == 0 or nj == m+1:
                        boundary_neighbors += 1  # Fixed to 0
                    elif ni == 1 or ni == n or nj == 1 or nj == m:
                        border_neighbors += 1    # Fixed to 1
            
            # Higher weight for cells with more fixed neighbors (tighter constraints)
            fixed_ratio = (border_neighbors + boundary_neighbors) / 8.0
            constraint_weight = 0.4 + 0.6 * fixed_ratio
            
            G.add_node(f'life_rule_{i}_{j}', type=1, weight=constraint_weight)
    
    # Global constraints
    # Border constraint: All border cells must be alive
    border_cells = 2*n + 2*m - 4  # Number of border cells
    border_weight = min(1.0, border_cells / (n*m))  # Fraction of total cells
    G.add_node('border_constraint', type=1, weight=border_weight)
    
    # Boundary constraint: All boundary cells must be dead
    G.add_node('boundary_constraint', type=1, weight=0.3)
    
    # Optimization constraint: Maximize living cells
    G.add_node('maximize_cells', type=1, weight=0.6)
    
    # Edges: Cell participation in constraints
    for i in range(1, n+1):
        for j in range(1, m+1):
            cell_node = f'cell_{i}_{j}'
            life_rule_node = f'life_rule_{i}_{j}'
            
            # Cell participates in its own life rule (primary relationship)
            G.add_edge(cell_node, life_rule_node, weight=1.0)
            
            # Cell participates in neighbors' life rules (secondary relationships)
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_rule = f'life_rule_{ni}_{nj}'
                        # Weight by proximity to border (cells near border have stronger influence)
                        influence = 0.3 + 0.4 * (1.0 - min(ni-1, n-ni, nj-1, m-nj) / max(n, m))
                        G.add_edge(cell_node, neighbor_rule, weight=influence)
            
            # Border cells participate in border constraint
            if i == 1 or i == n or j == 1 or j == m:
                G.add_edge(cell_node, 'border_constraint', weight=1.0)
            
            # All cells participate in maximization
            # Weight by potential contribution (corner cells worth more due to constraints)
            if (i == 1 or i == n) and (j == 1 or j == m):
                contribution = 1.0  # Corner cells
            elif i == 1 or i == n or j == 1 or j == m:
                contribution = 0.8  # Edge cells
            else:
                contribution = 0.6  # Interior cells
            
            G.add_edge(cell_node, 'maximize_cells', weight=contribution)
    
    # Add conflict edges for cells that strongly constrain each other
    # Focus on adjacent cells where Game of Life rules create tension
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Check 4-connected neighbors for potential conflicts
            for di, dj in [(0, 1), (1, 0)]:  # Only check right and down to avoid duplicates
                ni, nj = i + di, j + dj
                if 1 <= ni <= n and 1 <= nj <= m:
                    # Conflict strength based on constraint coupling
                    # Adjacent cells in Game of Life have strong mutual influence
                    cell1 = f'cell_{i}_{j}'
                    cell2 = f'cell_{ni}_{nj}'
                    
                    # Higher conflict near borders where cells have less freedom
                    border_proximity = 1.0 - min(i-1, n-i, j-1, m-j, ni-1, n-ni, nj-1, m-nj) / max(n, m)
                    conflict_weight = 0.2 + 0.3 * border_proximity
                    
                    G.add_edge(cell1, cell2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()