#!/usr/bin/env python3
"""
Graph converter for still_life_no_border problem.
Created using subagent_prompt.md version: v_02

This problem is about finding the maximum number of live cells in Conway's Game of Life
that form a stable configuration (still life) within an n×m grid without border cells.
Key challenges: balancing cell density with stability constraints, neighbor sum calculations.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the still life problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with cells as variables and constraints as separate nodes
    - Cell nodes (type 0): Each grid position, weighted by centrality and neighbor count
    - Constraint nodes (type 1): Neighbor sum constraints, stability rules, border constraints
    - What are the key entities? Grid cells and their neighbor relationships
    - What relationships matter for solving? Neighbor sum calculations, stability rules
    - What makes instances hard? Grid size, density vs stability tradeoffs
    """
    n = json_data.get('n', 3)  # number of rows
    m = json_data.get('m', 3)  # number of columns
    
    G = nx.Graph()
    
    # Cell nodes (type 0) - decision variables for whether each cell is alive
    max_neighbors = 8  # maximum possible neighbors for interior cells
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            # Calculate how central the cell is (more central = more constrained)
            dist_from_center_i = abs(i - (n + 1) / 2)
            dist_from_center_j = abs(j - (m + 1) / 2)
            centrality = 1.0 - (dist_from_center_i + dist_from_center_j) / (n + m)
            
            # Calculate number of potential neighbors (cells near border have fewer)
            neighbor_count = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_count += 1
            
            # Weight combines centrality and constraint complexity
            neighbor_factor = neighbor_count / max_neighbors
            weight = 0.3 * centrality + 0.7 * neighbor_factor
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=weight)
    
    # Constraint nodes (type 1) for different types of constraints
    
    # 1. Neighbor sum constraints - one for each cell
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            # Weight by complexity - more neighbors = more complex constraint
            neighbor_count = 0
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_count += 1
            
            constraint_weight = neighbor_count / max_neighbors
            G.add_node(f'sum_constraint_{i}_{j}', type=1, weight=constraint_weight)
    
    # 2. Stability constraints - separate for live and dead cell rules
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            # Live cell stability: if alive, must have 2 or 3 neighbors
            G.add_node(f'live_stability_{i}_{j}', type=1, weight=0.8)
            # Dead cell stability: if dead, must not have exactly 3 neighbors
            G.add_node(f'dead_stability_{i}_{j}', type=1, weight=0.7)
    
    # 3. Border constraints - ensure border cells are dead
    border_constraint_count = 0
    for i in [0, n + 1]:
        for j in range(m + 2):
            border_constraint_count += 1
    for j in [0, m + 1]:
        for i in range(1, n + 1):
            border_constraint_count += 1
    
    # Single border constraint node with weight based on border complexity
    border_weight = min(1.0, border_constraint_count / (2 * (n + m)))
    G.add_node('border_constraint', type=1, weight=border_weight)
    
    # 4. Global optimization constraint - maximizing live cells
    grid_size = n * m
    optimization_weight = math.log(grid_size + 1) / math.log(100)  # Non-linear scaling
    G.add_node('maximize_cost', type=1, weight=optimization_weight)
    
    # Edges: Connect cells to their relevant constraints
    
    # Connect cells to their neighbor sum constraints
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            cell = f'cell_{i}_{j}'
            sum_constraint = f'sum_constraint_{i}_{j}'
            
            # Primary connection - cell participates in its own sum constraint
            G.add_edge(cell, sum_constraint, weight=1.0)
            
            # Connect to neighbor cells' sum constraints (lower weight)
            for di in [-1, 0, 1]:
                for dj in [-1, 0, 1]:
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 1 <= ni <= n and 1 <= nj <= m:
                        neighbor_sum_constraint = f'sum_constraint_{ni}_{nj}'
                        # Distance-based weight (closer = stronger influence)
                        distance = math.sqrt(di*di + dj*dj)
                        weight = math.exp(-distance)  # Exponential decay
                        G.add_edge(cell, neighbor_sum_constraint, weight=weight)
    
    # Connect cells to stability constraints
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            cell = f'cell_{i}_{j}'
            live_stability = f'live_stability_{i}_{j}'
            dead_stability = f'dead_stability_{i}_{j}'
            
            G.add_edge(cell, live_stability, weight=0.9)
            G.add_edge(cell, dead_stability, weight=0.9)
    
    # Connect all cells to border constraint (they all must respect border being dead)
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            cell = f'cell_{i}_{j}'
            # Border cells more strongly connected
            min_border_dist = min(i - 1, j - 1, n - i, m - j)
            border_influence = math.exp(-min_border_dist)  # Exponential decay from border
            G.add_edge(cell, 'border_constraint', weight=border_influence)
    
    # Connect all cells to optimization constraint
    for i in range(1, n + 1):
        for j in range(1, m + 1):
            cell = f'cell_{i}_{j}'
            # Weight by potential contribution to objective
            G.add_edge(cell, 'maximize_cost', weight=0.6)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()