#!/usr/bin/env python3
"""
Graph converter for stripboard problem.
Created using subagent_prompt.md version: v_02

This problem is about laying out electrical components on a stripboard to minimize board area.
Key challenges: component placement, electrical routing, constraint satisfaction for physical layout.
The problem involves placing components with different footprints, managing pin connections through nets,
and potentially using jumper links to connect different rows.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path
import re


def parse_dzn_arrays(dzn_file):
    """
    Parse additional arrays from DZN file that may not be in JSON.
    This is a fallback for incomplete JSON conversion.
    """
    arrays = {}
    try:
        with open(dzn_file, 'r') as f:
            content = f.read()
        
        # Parse footprint_w array
        match = re.search(r'footprint_w\s*=\s*\[(.*?)\];', content, re.DOTALL)
        if match:
            footprint_w = {}
            items = match.group(1).split(',')
            for item in items:
                parts = item.strip().split(':')
                if len(parts) == 2:
                    comp = parts[0].strip()
                    val = int(parts[1].strip())
                    footprint_w[comp] = val
            arrays['footprint_w'] = footprint_w
        
        # Parse footprint_h array
        match = re.search(r'footprint_h\s*=\s*\[(.*?)\];', content, re.DOTALL)
        if match:
            footprint_h = {}
            items = match.group(1).split(',')
            for item in items:
                parts = item.strip().split(':')
                if len(parts) == 2:
                    comp = parts[0].strip()
                    val = int(parts[1].strip())
                    footprint_h[comp] = val
            arrays['footprint_h'] = footprint_h
        
        # Parse pin_net array
        match = re.search(r'pin_net\s*=\s*\[(.*?)\];', content, re.DOTALL)
        if match:
            pin_net = {}
            items = match.group(1).split(',')
            for item in items:
                parts = item.strip().split(':')
                if len(parts) == 2:
                    pin = parts[0].strip()
                    net = parts[1].strip()
                    pin_net[pin] = net
            arrays['pin_net'] = pin_net
        
        # Parse pins array (component to pin mappings)
        match = re.search(r'pins\s*=\s*\[(.*?)\];', content, re.DOTALL)
        if match:
            pins = {}
            items = match.group(1).split(',')
            for item in items:
                parts = item.strip().split(':')
                if len(parts) == 2:
                    comp = parts[0].strip()
                    pin_range = parts[1].strip()
                    pins[comp] = pin_range
            arrays['pins'] = pins
        
    except Exception as e:
        print(f"Warning: Could not parse DZN arrays: {e}")
    
    return arrays


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the stripboard layout problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the electrical layout problem
    - Variable nodes (type 0): Components and pins
    - Constraint nodes (type 1): Placement constraints, electrical nets, area constraints
    - Resource nodes (type 2): Board space regions
    
    The graph captures:
    - Component placement challenges (footprint sizes)
    - Electrical connectivity requirements (nets)
    - Physical layout constraints (non-overlap, board size)
    """
    
    # Get basic parameters
    max_w = json_data.get('max_w', 20)
    max_h = json_data.get('max_h', 10)
    max_links = json_data.get('max_links', 6)
    
    # Try to get additional data from DZN file
    dzn_arrays = parse_dzn_arrays(sys.argv[2]) if len(sys.argv) > 2 else {}
    
    # Estimate number of components and pins from JSON arrays
    num_components = len(json_data.get('COMPONENT', []))
    num_pins = len(json_data.get('PIN', []))
    num_nets = len(json_data.get('NET', []))
    
    G = nx.Graph()
    
    # Component nodes (type 0) - weighted by estimated layout difficulty
    for i in range(num_components):
        comp_id = f'comp_{i}'
        
        # Get footprint info if available
        footprint_w = dzn_arrays.get('footprint_w', {})
        footprint_h = dzn_arrays.get('footprint_h', {})
        
        # Estimate component difficulty based on size (larger = harder to place)
        if footprint_w and footprint_h:
            # Use actual footprint data
            comp_keys = list(footprint_w.keys())
            if i < len(comp_keys):
                comp_key = comp_keys[i]
                w = footprint_w.get(comp_key, 1)
                h = footprint_h.get(comp_key, 1)
                area = w * h
                # Larger components are harder to place
                weight = min(area / (max_w * max_h), 1.0)
            else:
                weight = 0.5
        else:
            # Fallback: vary by position
            weight = 0.3 + 0.4 * (i / max(num_components - 1, 1))
        
        G.add_node(comp_id, type=0, weight=weight)
    
    # Pin nodes (type 0) - weighted by connectivity complexity
    for i in range(num_pins):
        pin_id = f'pin_{i}'
        
        # Pins with more connections are more critical
        # Use position-based heuristic since we lack full connectivity info
        connectivity_weight = 0.2 + 0.6 * (i % 3) / 2.0  # Vary between 0.2-0.8
        
        G.add_node(pin_id, type=0, weight=connectivity_weight)
    
    # Net constraint nodes (type 1) - electrical connectivity requirements
    for i in range(num_nets):
        net_id = f'net_{i}'
        
        # Estimate net complexity (more pins = harder to route)
        # Use simple heuristics since we lack full pin-net mapping
        estimated_pins_per_net = max(num_pins // num_nets, 1)
        complexity = min(estimated_pins_per_net / 5.0, 1.0)  # Normalize to [0,1]
        
        G.add_node(net_id, type=1, weight=complexity)
    
    # Placement constraint nodes (type 1) - non-overlap and area constraints
    # Component placement constraints
    for i in range(num_components):
        placement_id = f'placement_{i}'
        
        # Larger board utilization makes placement harder
        utilization = (num_components * 3) / (max_w * max_h)  # Estimate area usage
        constraint_tightness = min(utilization * 1.5, 1.0)
        
        G.add_node(placement_id, type=1, weight=constraint_tightness)
    
    # Area optimization constraint
    G.add_node('area_constraint', type=1, weight=0.9)  # High weight - critical constraint
    
    # Board space resource nodes (type 2) - physical layout regions
    # Divide board into regions
    regions_per_dim = 3
    for row in range(regions_per_dim):
        for col in range(regions_per_dim):
            region_id = f'region_{row}_{col}'
            
            # Central regions are more valuable (higher contention)
            center_row, center_col = regions_per_dim // 2, regions_per_dim // 2
            distance_from_center = abs(row - center_row) + abs(col - center_col)
            scarcity = 1.0 - (distance_from_center / (regions_per_dim * 2))
            
            G.add_node(region_id, type=2, weight=scarcity)
    
    # Jumper link resources (type 2) if available
    if max_links > 0:
        for i in range(max_links):
            link_id = f'link_{i}'
            # Links are scarce resources
            G.add_node(link_id, type=2, weight=0.8)
    
    # Edges: Component-placement constraint relationships
    for i in range(num_components):
        comp_id = f'comp_{i}'
        placement_id = f'placement_{i}'
        
        # Strong relationship - component must satisfy placement constraint
        G.add_edge(comp_id, placement_id, weight=0.9)
        
        # Component affects area constraint
        G.add_edge(comp_id, 'area_constraint', weight=0.7)
    
    # Edges: Pin-net relationships
    for i in range(num_pins):
        pin_id = f'pin_{i}'
        # Connect each pin to estimated net (simple round-robin)
        net_idx = i % num_nets
        net_id = f'net_{net_idx}'
        
        # Strong electrical connectivity requirement
        G.add_edge(pin_id, net_id, weight=0.8)
    
    # Edges: Component-region relationships (spatial)
    for i in range(num_components):
        comp_id = f'comp_{i}'
        
        # Each component potentially uses multiple regions
        primary_region = i % (regions_per_dim * regions_per_dim)
        row = primary_region // regions_per_dim
        col = primary_region % regions_per_dim
        
        # Primary region
        region_id = f'region_{row}_{col}'
        G.add_edge(comp_id, region_id, weight=0.6)
        
        # Adjacent regions (lower weight)
        for dr, dc in [(-1,0), (1,0), (0,-1), (0,1)]:
            adj_row, adj_col = row + dr, col + dc
            if 0 <= adj_row < regions_per_dim and 0 <= adj_col < regions_per_dim:
                adj_region_id = f'region_{adj_row}_{adj_col}'
                G.add_edge(comp_id, adj_region_id, weight=0.3)
    
    # Edges: Pin-component relationships
    estimated_pins_per_comp = max(num_pins // num_components, 1)
    for i in range(num_components):
        comp_id = f'comp_{i}'
        
        # Connect component to its pins
        start_pin = i * estimated_pins_per_comp
        end_pin = min((i + 1) * estimated_pins_per_comp, num_pins)
        
        for pin_idx in range(start_pin, end_pin):
            pin_id = f'pin_{pin_idx}'
            # Pin belongs to component
            G.add_edge(comp_id, pin_id, weight=0.5)
    
    # Edges: Net-link relationships (if jumper links are used)
    if max_links > 0:
        for i in range(min(num_nets, max_links)):
            net_id = f'net_{i}'
            link_id = f'link_{i}'
            # Net might use jumper link for routing
            G.add_edge(net_id, link_id, weight=0.4)
    
    # Add some conflict edges for overlapping spatial constraints
    # Components that might compete for the same region
    for i in range(num_components):
        for j in range(i + 1, min(i + 3, num_components)):  # Check nearby components
            comp1_id = f'comp_{i}'
            comp2_id = f'comp_{j}'
            
            # Spatial conflict probability decreases with distance
            distance = abs(i - j)
            conflict_strength = math.exp(-distance / 2.0)  # Exponential decay
            
            if conflict_strength > 0.1:  # Only add meaningful conflicts
                G.add_edge(comp1_id, comp2_id, weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")
    
    # Print some statistics for verification
    node_types = {}
    for node, attrs in G.nodes(data=True):
        node_type = attrs.get('type', -1)
        node_types[node_type] = node_types.get(node_type, 0) + 1
    
    print(f"Node types: {node_types}")
    
    if G.number_of_edges() > 0:
        weights = [data['weight'] for _, _, data in G.edges(data=True)]
        print(f"Edge weight range: {min(weights):.3f} - {max(weights):.3f}")


if __name__ == "__main__":
    main()