#!/usr/bin/env python3
"""
Graph converter for Sudoku optimization problem.
Created using subagent_prompt.md version: v_02

This problem is about solving Sudoku puzzles while optimizing an objective function.
Key challenges: Constraint propagation in alldifferent constraints across rows, columns, and regions.
The optimization aspect adds complexity as we're not just satisfying constraints but finding optimal solutions.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Sudoku optimization problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with:
    - Cell nodes (type 0) representing decision variables
    - Constraint nodes (type 1) for rows, columns, and regions
    - Weights based on constraint complexity and cell importance
    """
    # Access data from json_data dict
    n = json_data.get('n', 9)  # Default to 9x9 if not specified
    x_values = json_data.get('x', [])  # Solution values (may be complete solution)
    
    # Calculate region size
    reg = math.ceil(math.sqrt(n))
    
    # Create graph
    G = nx.Graph()
    
    # Add cell nodes (type 0) - these are the decision variables
    for i in range(n):
        for j in range(n):
            cell_id = f'cell_{i}_{j}'
            
            # Calculate centrality weight - central cells are more constrained
            # because they participate in more conflicts
            center_dist = abs(i - n//2) + abs(j - n//2)
            max_dist = (n//2) + (n//2)
            centrality = 1.0 - (center_dist / max_dist) if max_dist > 0 else 1.0
            
            # Corner and edge effects - cells near boundaries have fewer neighbors
            edge_penalty = 0.0
            if i == 0 or i == n-1:
                edge_penalty += 0.1
            if j == 0 or j == n-1:
                edge_penalty += 0.1
                
            cell_weight = max(0.1, centrality - edge_penalty)
            G.add_node(cell_id, type=0, weight=cell_weight)
    
    # Add row constraint nodes (type 1)
    for i in range(n):
        constraint_id = f'row_{i}'
        # All row constraints have same scope (n variables)
        G.add_node(constraint_id, type=1, weight=1.0)
        
        # Connect all cells in this row to the row constraint
        for j in range(n):
            cell_id = f'cell_{i}_{j}'
            G.add_edge(cell_id, constraint_id, weight=1.0)
    
    # Add column constraint nodes (type 1)
    for j in range(n):
        constraint_id = f'col_{j}'
        # All column constraints have same scope (n variables)
        G.add_node(constraint_id, type=1, weight=1.0)
        
        # Connect all cells in this column to the column constraint
        for i in range(n):
            cell_id = f'cell_{i}_{j}'
            G.add_edge(cell_id, constraint_id, weight=1.0)
    
    # Add region constraint nodes (type 1)
    for reg_i in range(reg):
        for reg_j in range(reg):
            constraint_id = f'region_{reg_i}_{reg_j}'
            
            # Count actual cells in this region (may be fewer for irregular grids)
            cells_in_region = []
            for r in range(reg_i * reg, min((reg_i + 1) * reg, n)):
                for c in range(reg_j * reg, min((reg_j + 1) * reg, n)):
                    cells_in_region.append((r, c))
            
            # Weight by region size (smaller regions are less constraining)
            region_size = len(cells_in_region)
            max_region_size = reg * reg
            region_weight = region_size / max_region_size if max_region_size > 0 else 1.0
            
            G.add_node(constraint_id, type=1, weight=region_weight)
            
            # Connect all cells in this region to the region constraint
            for r, c in cells_in_region:
                cell_id = f'cell_{r}_{c}'
                G.add_edge(cell_id, constraint_id, weight=1.0)
    
    # Add objective constraint node (type 1) - the optimization objective
    G.add_node('objective_constraint', type=1, weight=0.8)
    
    # Connect cells to objective with weights based on their contribution
    # The objective is: sum([x[i,j]*(-1)^(i+j) | i,j in 1..n])
    for i in range(n):
        for j in range(n):
            cell_id = f'cell_{i}_{j}'
            # Weight based on contribution to objective (checkerboard pattern)
            # Cells with positive contribution are more important
            coefficient = (-1) ** (i + j)  # This matches the MiniZinc formula
            obj_weight = 0.5 + 0.3 * abs(coefficient)  # Range [0.2, 0.8]
            G.add_edge(cell_id, 'objective_constraint', weight=obj_weight)
    
    # Add high-interaction edges between cells that share multiple constraints
    # This captures the complex constraint interactions in Sudoku
    for i1 in range(n):
        for j1 in range(n):
            for i2 in range(i1, n):
                for j2 in range(n):
                    if i1 == i2 and j1 >= j2:  # Skip self and avoid duplicates
                        continue
                        
                    cell1 = f'cell_{i1}_{j1}'
                    cell2 = f'cell_{i2}_{j2}'
                    
                    # Count shared constraints
                    shared_constraints = 0
                    
                    # Same row
                    if i1 == i2:
                        shared_constraints += 1
                    
                    # Same column  
                    if j1 == j2:
                        shared_constraints += 1
                    
                    # Same region
                    if (i1 // reg == i2 // reg) and (j1 // reg == j2 // reg):
                        shared_constraints += 1
                    
                    # Add conflict edge for cells sharing 2+ constraints
                    if shared_constraints >= 2:
                        # Weight by number of shared constraints (more = stronger conflict)
                        conflict_weight = shared_constraints / 3.0  # Max 3 constraints can be shared
                        # Use exponential scaling for stronger signal
                        conflict_weight = min(1.0, math.exp(conflict_weight - 1))
                        G.add_edge(cell1, cell2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()