#!/usr/bin/env python3
"""
Graph converter for sugiyama2-mznchallenge2010 problem.
Created using subagent_prompt.md version: v_02

This problem is about optimal ordering layout for Sugiyama-style graphs to minimize edge crossings.
Key challenges: Complex edge crossing patterns, layer-dependent constraints, positioning conflicts.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Sugiyama layout problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph capturing layer structure and crossing complexity
    - Node positions as variables (type 0) with crossing-based weights
    - Layer constraints (type 1) weighted by width and connectivity density
    - Edge crossings as constraint relationships with crossing potential weights
    - Inter-layer edges modeled with exponential distance decay
    """
    layers = json_data.get('layers', 0)
    nodes = json_data.get('nodes', 0)
    edges_count = json_data.get('edges', 0)
    width = json_data.get('width', [])
    start = json_data.get('start', [])
    end = json_data.get('end', [])
    
    if not all([layers, nodes, edges_count, width, start, end]):
        # Return minimal graph for empty instances
        G = nx.Graph()
        G.add_node('dummy', type=0, weight=0.5)
        return G
    
    G = nx.Graph()
    
    # Calculate layer boundaries for node assignment
    layer_boundaries = [0]
    for w in width:
        layer_boundaries.append(layer_boundaries[-1] + w)
    
    # Determine which layer each node belongs to
    node_to_layer = {}
    for layer_idx in range(layers):
        start_node = layer_boundaries[layer_idx] + 1
        end_node = layer_boundaries[layer_idx + 1]
        for node_id in range(start_node, end_node + 1):
            node_to_layer[node_id] = layer_idx
    
    # Count incoming and outgoing edges for each node
    node_in_degree = {i: 0 for i in range(1, nodes + 1)}
    node_out_degree = {i: 0 for i in range(1, nodes + 1)}
    
    for i in range(edges_count):
        if i < len(start) and i < len(end):
            start_node = start[i]
            end_node = end[i]
            if start_node in node_out_degree:
                node_out_degree[start_node] += 1
            if end_node in node_in_degree:
                node_in_degree[end_node] += 1
    
    # Add variable nodes (node positions) with crossing-potential weights
    max_degree = max(max(node_in_degree.values(), default=1), 
                     max(node_out_degree.values(), default=1))
    
    for node_id in range(1, nodes + 1):
        layer_idx = node_to_layer.get(node_id, 0)
        layer_width = width[layer_idx] if layer_idx < len(width) else 1
        
        # Weight based on connectivity and central position influence
        in_deg = node_in_degree.get(node_id, 0)
        out_deg = node_out_degree.get(node_id, 0)
        total_deg = in_deg + out_deg
        
        # Nodes with higher degree have more crossing potential
        degree_weight = total_deg / (2 * max_degree) if max_degree > 0 else 0.5
        
        # Nodes in wider layers have more positioning flexibility
        layer_complexity = math.log(layer_width + 1) / math.log(max(width) + 1) if width else 0.5
        
        # Combine factors with non-linear scaling
        weight = 0.3 + 0.7 * (degree_weight * 0.7 + layer_complexity * 0.3)
        weight = min(weight, 1.0)
        
        G.add_node(f'pos_{node_id}', type=0, weight=weight)
    
    # Add constraint nodes for each layer (all_different constraints)
    for layer_idx in range(layers):
        layer_width = width[layer_idx] if layer_idx < len(width) else 1
        
        # Count edges within and crossing this layer
        layer_nodes = []
        for node_id in range(1, nodes + 1):
            if node_to_layer.get(node_id, -1) == layer_idx:
                layer_nodes.append(node_id)
        
        # Calculate crossing complexity for this layer
        crossing_edges = 0
        for i in range(edges_count):
            if i < len(start) and i < len(end):
                start_node = start[i]
                end_node = end[i]
                start_layer = node_to_layer.get(start_node, -1)
                end_layer = node_to_layer.get(end_node, -1)
                
                # Count edges originating from this layer
                if start_layer == layer_idx and end_layer != layer_idx:
                    crossing_edges += 1
        
        # Layer constraint weight based on width and crossing potential
        width_factor = layer_width / max(width) if width else 0.5
        crossing_factor = crossing_edges / max(edges_count, 1)
        
        # Non-linear combination emphasizing crossing complexity
        weight = 0.4 + 0.6 * math.sqrt(width_factor * 0.6 + crossing_factor * 0.4)
        
        G.add_node(f'layer_constraint_{layer_idx}', type=1, weight=weight)
    
    # Add crossing constraint nodes for edge pairs that can cross
    crossing_constraints = 0
    max_crossings_per_layer = 50  # Limit to prevent explosion
    
    for layer_idx in range(layers - 1):  # Between adjacent layers
        layer_edges = []
        for i in range(edges_count):
            if i < len(start) and i < len(end):
                start_node = start[i]
                start_layer = node_to_layer.get(start_node, -1)
                if start_layer == layer_idx:
                    layer_edges.append(i)
        
        # Add crossing constraints for edge pairs in this layer
        added_this_layer = 0
        for i in range(len(layer_edges)):
            for j in range(i + 1, len(layer_edges)):
                if added_this_layer >= max_crossings_per_layer:
                    break
                
                edge1_idx = layer_edges[i]
                edge2_idx = layer_edges[j]
                
                if edge1_idx < len(start) and edge1_idx < len(end) and \
                   edge2_idx < len(start) and edge2_idx < len(end):
                    
                    # Check if edges can actually cross
                    s1, e1 = start[edge1_idx], end[edge1_idx]
                    s2, e2 = start[edge2_idx], end[edge2_idx]
                    
                    if s1 != s2 and e1 != e2:  # Different start and end nodes
                        # Weight based on potential for crossing
                        # Higher weight for edges with more potential crossing conflicts
                        s1_deg = node_in_degree.get(s1, 0) + node_out_degree.get(s1, 0)
                        s2_deg = node_in_degree.get(s2, 0) + node_out_degree.get(s2, 0)
                        
                        crossing_potential = (s1_deg + s2_deg) / (4 * max_degree) if max_degree > 0 else 0.5
                        weight = 0.2 + 0.8 * math.exp(-2.0 * (1.0 - crossing_potential))
                        
                        G.add_node(f'crossing_{crossing_constraints}', type=1, weight=weight)
                        crossing_constraints += 1
                        added_this_layer += 1
            
            if added_this_layer >= max_crossings_per_layer:
                break
    
    # Add edges: variables to layer constraints (bipartite)
    for node_id in range(1, nodes + 1):
        layer_idx = node_to_layer.get(node_id, -1)
        if layer_idx >= 0:
            # Edge weight based on node's influence on layer complexity
            degree = node_in_degree.get(node_id, 0) + node_out_degree.get(node_id, 0)
            influence = degree / (2 * max_degree) if max_degree > 0 else 0.5
            edge_weight = 0.4 + 0.6 * influence
            
            G.add_edge(f'pos_{node_id}', f'layer_constraint_{layer_idx}', weight=edge_weight)
    
    # Add edges: variables to crossing constraints
    crossing_idx = 0
    for layer_idx in range(layers - 1):
        layer_edges = []
        for i in range(edges_count):
            if i < len(start) and i < len(end):
                start_node = start[i]
                start_layer = node_to_layer.get(start_node, -1)
                if start_layer == layer_idx:
                    layer_edges.append((i, start_node, end[i]))
        
        added_this_layer = 0
        for i in range(len(layer_edges)):
            for j in range(i + 1, len(layer_edges)):
                if added_this_layer >= max_crossings_per_layer or crossing_idx >= crossing_constraints:
                    break
                
                _, s1, e1 = layer_edges[i]
                _, s2, e2 = layer_edges[j]
                
                if s1 != s2 and e1 != e2:
                    # Connect participating nodes to crossing constraint
                    crossing_node = f'crossing_{crossing_idx}'
                    
                    # Weight based on participation strength in potential crossing
                    participation_weight = 0.6 + 0.4 * math.sqrt(
                        (node_out_degree.get(s1, 0) + node_out_degree.get(s2, 0)) / 
                        (2 * max_degree) if max_degree > 0 else 0.5
                    )
                    
                    G.add_edge(f'pos_{s1}', crossing_node, weight=participation_weight)
                    G.add_edge(f'pos_{s2}', crossing_node, weight=participation_weight)
                    G.add_edge(f'pos_{e1}', crossing_node, weight=participation_weight * 0.8)
                    G.add_edge(f'pos_{e2}', crossing_node, weight=participation_weight * 0.8)
                    
                    crossing_idx += 1
                    added_this_layer += 1
            
            if added_this_layer >= max_crossings_per_layer:
                break
    
    # Add direct edges between adjacent layer nodes with high connectivity
    for i in range(edges_count):
        if i < len(start) and i < len(end):
            start_node = start[i]
            end_node = end[i]
            start_layer = node_to_layer.get(start_node, -1)
            end_layer = node_to_layer.get(end_node, -1)
            
            # Add edge for direct graph connections with distance decay
            if start_layer != end_layer and start_layer >= 0 and end_layer >= 0:
                layer_distance = abs(end_layer - start_layer)
                
                # Exponential decay with layer distance
                distance_weight = math.exp(-1.5 * layer_distance / layers) if layers > 0 else 0.5
                
                # Boost weight for high-degree nodes
                start_deg = node_in_degree.get(start_node, 0) + node_out_degree.get(start_node, 0)
                end_deg = node_in_degree.get(end_node, 0) + node_out_degree.get(end_node, 0)
                degree_boost = math.sqrt((start_deg + end_deg) / (4 * max_degree)) if max_degree > 0 else 0.5
                
                final_weight = distance_weight * (0.4 + 0.6 * degree_boost)
                
                G.add_edge(f'pos_{start_node}', f'pos_{end_node}', weight=final_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()