#!/usr/bin/env python3
"""
Graph converter for Time-changing Graph Coloring Problem (tcgc2).
Created using subagent_prompt.md version: v_02

This problem is about transforming one valid graph coloring into another through 
a sequence of steps with at most k color changes per step.
Key challenges: minimizing transformation steps while maintaining valid colorings
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the time-changing graph coloring problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the problem as a bipartite graph capturing:
    - Variables: nodes in the original graph (type 0)
    - Constraints: valid coloring constraints for edges (type 1)
    - Resource: transformation capacity k (type 2)
    - Transformation difficulty based on graph structure and change requirements
    """
    # Extract graph structure
    nbV = json_data.get('nbV', 0)
    nbE = json_data.get('nbE', 0)
    tails = json_data.get('tails', [])
    heads = json_data.get('heads', [])
    k = json_data.get('k', 1)  # max changes per step
    
    # Note: init and end colorings are missing from JSON conversion
    # This is a known limitation with array1d parsing
    
    G = nx.Graph()
    
    if nbV == 0 or nbE == 0:
        return G
    
    # Calculate node degrees to assess constraint complexity
    node_degrees = [0] * (nbV + 1)  # 1-indexed
    for i in range(min(len(tails), len(heads))):
        if i < len(tails) and i < len(heads):
            tail = tails[i]
            head = heads[i]
            if 1 <= tail <= nbV:
                node_degrees[tail] += 1
            if 1 <= head <= nbV:
                node_degrees[head] += 1
    
    max_degree = max(node_degrees[1:]) if nbV > 0 else 1
    
    # Add variable nodes (graph vertices that need coloring)
    for v in range(1, nbV + 1):
        # Weight based on degree - higher degree nodes are more constrained
        degree = node_degrees[v]
        constraint_weight = degree / max_degree if max_degree > 0 else 0.5
        # Use exponential weighting to emphasize high-degree nodes
        weight = 1.0 - math.exp(-3.0 * constraint_weight)
        G.add_node(f'vertex_{v}', type=0, weight=weight)
    
    # Add constraint nodes for each edge (valid coloring constraints)
    for e in range(min(len(tails), len(heads))):
        if e < len(tails) and e < len(heads):
            tail = tails[e]
            head = heads[e]
            if 1 <= tail <= nbV and 1 <= head <= nbV:
                # Weight based on how constrained the incident vertices are
                tail_degree = node_degrees[tail]
                head_degree = node_degrees[head]
                avg_degree = (tail_degree + head_degree) / 2
                constraint_tightness = avg_degree / max_degree if max_degree > 0 else 0.5
                # Higher weight for constraints involving high-degree vertices
                weight = math.sqrt(constraint_tightness)
                G.add_node(f'edge_constraint_{e}', type=1, weight=weight)
    
    # Add resource node representing transformation capacity
    # Weight based on how restrictive k is relative to graph size
    transformation_difficulty = 1.0 - (k / nbV) if nbV > 0 else 0.5
    transformation_difficulty = max(0.1, min(1.0, transformation_difficulty))
    G.add_node('transformation_capacity', type=2, weight=transformation_difficulty)
    
    # Add bipartite edges: variables to their incident edge constraints
    for e in range(min(len(tails), len(heads))):
        if e < len(tails) and e < len(heads):
            tail = tails[e]
            head = heads[e]
            if 1 <= tail <= nbV and 1 <= head <= nbV:
                constraint_node = f'edge_constraint_{e}'
                
                # Edge weight based on how critical this constraint is
                tail_degree = node_degrees[tail]
                head_degree = node_degrees[head]
                criticality = (tail_degree + head_degree) / (2 * max_degree) if max_degree > 0 else 0.5
                edge_weight = math.sqrt(criticality)
                
                G.add_edge(f'vertex_{tail}', constraint_node, weight=edge_weight)
                G.add_edge(f'vertex_{head}', constraint_node, weight=edge_weight)
    
    # Connect high-degree vertices to transformation capacity
    # These vertices are harder to change due to many constraints
    degree_threshold = max_degree * 0.7 if max_degree > 0 else 1
    for v in range(1, nbV + 1):
        if node_degrees[v] >= degree_threshold:
            # Weight based on how much this vertex limits transformation
            limitation = node_degrees[v] / max_degree if max_degree > 0 else 0.5
            G.add_edge(f'vertex_{v}', 'transformation_capacity', 
                      weight=limitation)
    
    # Add conflict edges between highly constrained adjacent vertices
    # These represent particularly difficult transformation bottlenecks
    for e in range(min(len(tails), len(heads))):
        if e < len(tails) and e < len(heads):
            tail = tails[e]
            head = heads[e]
            if (1 <= tail <= nbV and 1 <= head <= nbV and 
                node_degrees[tail] >= degree_threshold and 
                node_degrees[head] >= degree_threshold):
                
                # Conflict weight based on combined constraint pressure
                conflict_intensity = (node_degrees[tail] + node_degrees[head]) / (2 * max_degree)
                conflict_weight = math.exp(-2.0 * (1.0 - conflict_intensity))
                G.add_edge(f'vertex_{tail}', f'vertex_{head}', 
                          weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()