#!/usr/bin/env python3
"""
Graph converter for Time Dependent Traveling Salesman Problem (TDTSP).
Created using subagent_prompt.md version: v_02

This problem is about finding an optimal tour through n visits where travel times
between locations depend on the departure time. Each visit has a duration, and there
may be precedence constraints and forbidden time windows.

Key challenges: Time-dependent travel costs create complex interactions between
routing decisions and timing decisions. The precedence constraints and forbidden
windows add additional scheduling complexity.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the TDTSP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with visit nodes and constraint nodes.
    - Visit nodes (type 0): Each visit location with weight based on duration
    - Constraint nodes (type 1): Precedence constraints, forbidden windows, tour structure
    - Edges connect visits to constraints they participate in
    - Edge weights reflect constraint tightness and visit criticality
    """
    # Extract data from JSON
    n = json_data.get('n', 0)  # Number of visits
    D = json_data.get('D', [])  # Duration of each visit
    q = json_data.get('q', 0)  # Number of precedence constraints
    prec = json_data.get('prec', [])  # Precedence constraints
    r = json_data.get('r', 0)  # Number of forbidden regions
    interval = json_data.get('interval', [])  # Forbidden time intervals
    steps = json_data.get('steps', 64)  # Number of time steps
    l = json_data.get('l', 360)  # Travel time granularity
    
    if n == 0:
        return nx.Graph()
    
    G = nx.Graph()
    
    # Add visit nodes (type 0) - weight by relative duration importance
    max_duration = max(D) if D else 1
    min_duration = min(D) if D else 1
    duration_range = max_duration - min_duration
    
    for i in range(n):
        duration = D[i] if i < len(D) else 1
        # Longer durations are more constraining for scheduling
        if duration_range > 0:
            weight = (duration - min_duration) / duration_range
        else:
            weight = 0.5
        # Apply non-linear scaling to emphasize differences
        weight = 0.1 + 0.9 * math.sqrt(weight)
        G.add_node(f'visit_{i+1}', type=0, weight=weight)
    
    # Add tour structure constraint nodes (type 1)
    # Each visit must have exactly one predecessor and one successor
    for i in range(n):
        # Predecessor constraint for visit i+1
        G.add_node(f'pred_constraint_{i+1}', type=1, weight=1.0)
        # Successor constraint for visit i+1  
        G.add_node(f'succ_constraint_{i+1}', type=1, weight=1.0)
        
        # Connect visit to its ordering constraints
        G.add_edge(f'visit_{i+1}', f'pred_constraint_{i+1}', weight=1.0)
        G.add_edge(f'visit_{i+1}', f'succ_constraint_{i+1}', weight=1.0)
    
    # Add precedence constraint nodes (type 1)
    for i in range(q):
        if i < len(prec) and len(prec[i]) >= 2:
            before_visit = prec[i][0]
            after_visit = prec[i][1]
            
            constraint_id = f'precedence_{i}'
            # Precedence constraints are highly restrictive
            G.add_node(constraint_id, type=1, weight=0.9)
            
            # Connect the two visits involved in precedence
            if 1 <= before_visit <= n:
                G.add_edge(f'visit_{before_visit}', constraint_id, weight=0.8)
            if 1 <= after_visit <= n:
                G.add_edge(f'visit_{after_visit}', constraint_id, weight=0.8)
    
    # Add forbidden time window constraint nodes (type 1)
    time_horizon = steps * l - 1
    
    for i in range(r):
        if i < len(interval) and len(interval[i]) >= 2:
            start_time = interval[i][0] if len(interval[i]) > 0 else 0
            end_time = interval[i][1] if len(interval[i]) > 1 else time_horizon
            
            # Calculate constraint tightness based on forbidden window size
            window_size = end_time - start_time
            tightness = min(window_size / time_horizon, 1.0) if time_horizon > 0 else 0.5
            # Larger forbidden windows are more restrictive
            weight = 0.3 + 0.7 * tightness
            
            constraint_id = f'forbidden_{i}'
            G.add_node(constraint_id, type=1, weight=weight)
            
            # Connect to affected visit (would need 'which' array from DZN)
            # Since 'which' is not in JSON, connect to all visits with lower weight
            for visit_idx in range(1, n+1):
                G.add_edge(f'visit_{visit_idx}', constraint_id, weight=0.3)
    
    # Add time dependency complexity nodes (type 1)
    # These represent the additional complexity from time-dependent travel
    if n > 1:
        # Global time dependency constraint
        time_complexity = min(1.0, (steps * n) / 1000.0)  # Scale by problem size
        G.add_node('time_dependency', type=1, weight=0.5 + 0.5 * time_complexity)
        
        # Connect all visits to time dependency constraint
        # Visits with longer durations have stronger connection to time complexity
        for i in range(n):
            duration = D[i] if i < len(D) else 1
            duration_weight = (duration / max_duration) if max_duration > 0 else 0.5
            edge_weight = 0.2 + 0.6 * duration_weight
            G.add_edge(f'visit_{i+1}', 'time_dependency', weight=edge_weight)
    
    # Add travel time variability constraints (type 1)
    # These capture the complexity of time-dependent routing
    if n > 2:
        # Create constraints for groups of visits that compete for optimal timing
        for group_start in range(0, n, 3):  # Group visits in sets of 3
            group_end = min(group_start + 3, n)
            if group_end - group_start >= 2:
                group_id = f'timing_group_{group_start//3}'
                
                # Weight by group size (larger groups more complex)
                group_size = group_end - group_start
                weight = 0.4 + 0.4 * (group_size / 3.0)
                G.add_node(group_id, type=1, weight=weight)
                
                # Connect visits in this timing group
                for i in range(group_start, group_end):
                    visit_id = f'visit_{i+1}'
                    # Edge weight based on visit duration (longer = more impact on timing)
                    duration = D[i] if i < len(D) else 1
                    edge_weight = 0.3 + 0.5 * (duration / max_duration) if max_duration > 0 else 0.5
                    G.add_edge(visit_id, group_id, weight=edge_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()