#!/usr/bin/env python3
"""
Graph converter for Time-Dependent Traveling Salesman Problem (TDTSP).
Created using subagent_prompt.md version: v_02

This problem is about finding an optimal tour visiting all locations with 
time-dependent travel costs, duration constraints, precedence relationships,
and forbidden time windows.

Key challenges: Time-dependent travel costs, precedence constraints, 
forbidden time windows, duration requirements.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the TDTSP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with visit nodes and constraint nodes.
    - Visit nodes represent the locations to visit
    - Constraint nodes represent precedence, duration, and time window constraints
    - Weights capture difficulty: visit importance, constraint tightness
    - Time-dependent nature modeled through weighted connections
    """
    # Extract data from JSON
    n = json_data.get('n', 0)  # Number of visits
    D = json_data.get('D', [])  # Visit durations
    q = json_data.get('q', 0)  # Number of precedences
    prec = json_data.get('prec', [])  # Precedence constraints
    steps = json_data.get('steps', 1)  # Time steps
    l = json_data.get('l', 1)  # Time granularity
    r = json_data.get('r', 0)  # Number of forbidden regions
    
    # Calculate time horizon
    H = steps * l - 1
    
    # Create graph
    G = nx.Graph()
    
    # Add visit nodes (type 0 - variables)
    # Weight by duration complexity and time horizon pressure
    max_duration = max(D) if D else 1
    total_duration = sum(D) if D else 1
    
    for i in range(n):
        duration = D[i] if i < len(D) else 1
        # Longer durations are more constraining in scheduling
        duration_weight = duration / max_duration if max_duration > 0 else 0.5
        # Also consider relative impact on total time
        impact_weight = duration / total_duration if total_duration > 0 else 0.5
        # Combine with non-linear scaling for high-impact visits
        visit_weight = min(0.5 * duration_weight + 0.5 * math.exp(impact_weight) / math.e, 1.0)
        
        G.add_node(f'visit_{i+1}', type=0, weight=visit_weight)
    
    # Add depot nodes (start and end)
    G.add_node('depot_start', type=0, weight=0.8)  # High importance as tour start
    G.add_node('depot_end', type=0, weight=0.9)    # High importance as tour end
    
    # Add precedence constraint nodes (type 1)
    for i in range(q):
        if i < len(prec) and len(prec[i]) >= 2:
            # Precedence constraints create ordering dependencies
            # Weight by the number of visits affected and their durations
            before_visit = prec[i][0] if prec[i][0] <= n else 1
            after_visit = prec[i][1] if prec[i][1] <= n else 1
            
            # Consider impact of both visits involved
            before_duration = D[before_visit-1] if before_visit-1 < len(D) else 1
            after_duration = D[after_visit-1] if after_visit-1 < len(D) else 1
            combined_duration = before_duration + after_duration
            
            # Precedence constraints are more critical when they involve longer durations
            precedence_weight = min(combined_duration / (2 * max_duration), 1.0) if max_duration > 0 else 0.7
            
            G.add_node(f'precedence_{i}', type=1, weight=precedence_weight)
            
            # Connect precedence constraint to involved visits
            G.add_edge(f'visit_{before_visit}', f'precedence_{i}', weight=0.8)
            G.add_edge(f'visit_{after_visit}', f'precedence_{i}', weight=0.8)
    
    # Add forbidden time window constraint nodes (type 1)
    for i in range(r):
        # Forbidden windows create scheduling restrictions
        # Weight by window size and affected visit duration
        interval_data = json_data.get('interval', [])
        which_data = json_data.get('which', [])
        
        if i < len(interval_data) and len(interval_data[i]) >= 2:
            window_start = interval_data[i][0]
            window_end = interval_data[i][1]
            window_size = window_end - window_start
            
            # Larger forbidden windows are more constraining
            window_tightness = min(window_size / H, 1.0) if H > 0 else 0.5
            # Invert: smaller windows (more restrictive) get higher weights
            forbidden_weight = max(1.0 - window_tightness, 0.3)
            
            G.add_node(f'forbidden_{i}', type=1, weight=forbidden_weight)
            
            # Connect to affected visit
            if i < len(which_data):
                affected_visit = which_data[i]
                if 1 <= affected_visit <= n:
                    G.add_edge(f'visit_{affected_visit}', f'forbidden_{i}', weight=0.7)
    
    # Add time horizon constraint (type 1)
    # This represents the global scheduling pressure
    time_pressure = min(total_duration / H, 1.0) if H > 0 else 0.5
    G.add_node('time_horizon', type=1, weight=time_pressure)
    
    # Connect time horizon to all visits with weights based on their scheduling impact
    for i in range(n):
        duration = D[i] if i < len(D) else 1
        # Visits with longer durations put more pressure on the time horizon
        schedule_pressure = min(duration / max_duration, 1.0) if max_duration > 0 else 0.5
        G.add_edge(f'visit_{i+1}', 'time_horizon', weight=schedule_pressure)
    
    # Add tour structure constraints (type 1)
    # These represent the TSP tour requirements
    G.add_node('tour_structure', type=1, weight=0.8)
    
    # Connect tour structure to all visits and depot
    base_tour_weight = 0.6
    for i in range(n):
        # Visits that are harder to schedule get higher tour structure weights
        duration = D[i] if i < len(D) else 1
        tour_weight = base_tour_weight + 0.3 * (duration / max_duration) if max_duration > 0 else base_tour_weight
        G.add_edge(f'visit_{i+1}', 'tour_structure', weight=min(tour_weight, 1.0))
    
    G.add_edge('depot_start', 'tour_structure', weight=0.7)
    G.add_edge('depot_end', 'tour_structure', weight=0.7)
    
    # Add complexity-based resource node (type 2)
    # Represents the time resource which all activities compete for
    G.add_node('time_resource', type=2, weight=time_pressure)
    
    # Connect all visits to time resource with consumption-based weights
    for i in range(n):
        duration = D[i] if i < len(D) else 1
        # Time consumption relative to total available time
        time_consumption = duration / H if H > 0 else 0.5
        # Use logarithmic scaling for better differentiation
        if time_consumption > 0:
            consumption_weight = min(math.log(1 + time_consumption * 10) / math.log(11), 1.0)
        else:
            consumption_weight = 0.1
        
        G.add_edge(f'visit_{i+1}', 'time_resource', weight=consumption_weight)
    
    # Add problem complexity nodes based on scale
    # For larger problems, add additional structure
    if n >= 15:  # Larger instances
        # Add regional clustering constraint (type 1)
        G.add_node('regional_complexity', type=1, weight=0.6)
        
        # Connect to subset of visits (those with higher duration impact)
        for i in range(n):
            duration = D[i] if i < len(D) else 1
            if duration > 0.7 * max_duration:  # High-duration visits
                G.add_edge(f'visit_{i+1}', 'regional_complexity', weight=0.5)
    
    # Add cross-constraint interactions for complex instances
    if q > 0 and r > 0:  # Both precedences and forbidden windows exist
        G.add_node('constraint_interaction', type=1, weight=0.7)
        
        # Connect to precedence and forbidden window constraints
        for i in range(min(q, 3)):  # Connect to first few precedence constraints
            if G.has_node(f'precedence_{i}'):
                G.add_edge(f'precedence_{i}', 'constraint_interaction', weight=0.4)
        
        for i in range(min(r, 3)):  # Connect to first few forbidden constraints
            if G.has_node(f'forbidden_{i}'):
                G.add_edge(f'forbidden_{i}', 'constraint_interaction', weight=0.4)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()