#!/usr/bin/env python3
"""
Graph converter for Template Design problem.
Created using subagent_prompt.md version: v_02

This problem is about designing printing templates to minimize production while meeting demands.
Key challenges: slot allocation across templates, balancing template utilization, meeting demand constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the template design problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with variations and constraint nodes
    - Variations (type 0): products competing for template slots
    - Template capacity constraints (type 1): each template must be full (S slots)
    - Demand constraints (type 1): each variation must meet its demand
    - Template resources (type 2): represent template utilization pressure
    
    Weights reflect:
    - Variation importance: based on relative demand
    - Constraint tightness: how constraining each constraint is
    - Resource pressure: template utilization difficulty
    """
    # Access data directly from json_data dict
    S = json_data.get('S', 1)  # slots per template
    t = json_data.get('t', 1)  # number of templates
    n = json_data.get('n', 0)  # number of variations
    d = json_data.get('d', [])  # demands
    
    # Create graph
    G = nx.Graph()
    
    # Calculate derived metrics
    total_demand = sum(d) if d else 1
    max_demand = max(d) if d else 1
    total_slots = S * t
    
    # Calculate demand pressure (how tight the constraints are)
    demand_pressure = total_demand / total_slots if total_slots > 0 else 1.0
    
    # Add variation nodes (type 0) - decision variables
    for i in range(n):
        demand = d[i] if i < len(d) else 0
        # Weight by relative demand importance (non-linear scaling)
        demand_ratio = demand / max_demand if max_demand > 0 else 0.5
        # Use sqrt to give high-demand items more weight but not linearly
        weight = math.sqrt(demand_ratio)
        G.add_node(f'var_{i}', type=0, weight=weight)
    
    # Add template capacity constraint nodes (type 1)
    # Each template must have exactly S slots filled
    for j in range(t):
        # Template constraints are equally tight (must be exactly S)
        # Weight by relative template importance in problem
        template_importance = 1.0 / t  # Equal importance but decreasing with more templates
        G.add_node(f'template_cap_{j}', type=1, weight=template_importance)
    
    # Add demand constraint nodes (type 1)  
    # Each variation must meet its demand
    for i in range(n):
        demand = d[i] if i < len(d) else 0
        # Tightness based on how much of total capacity this demand requires
        relative_demand = demand / total_slots if total_slots > 0 else 0.5
        # Higher demand = tighter constraint
        tightness = min(relative_demand * 2.0, 1.0)  # Cap at 1.0
        G.add_node(f'demand_{i}', type=1, weight=tightness)
    
    # Add template resource nodes (type 2)
    # Represent the limited template resources
    for j in range(t):
        # Resource scarcity: fewer templates = more scarce
        scarcity = min(demand_pressure, 1.0)  # How pressured this resource is
        G.add_node(f'template_res_{j}', type=2, weight=scarcity)
    
    # Add participation edges: variation -> template capacity constraints
    # Each variation can be assigned to any template
    for i in range(n):
        for j in range(t):
            # Edge weight reflects how much this variation "wants" this template
            demand = d[i] if i < len(d) else 0
            slot_pressure = demand / S if S > 0 else 0.5  # How many slots this var might need
            participation_weight = min(slot_pressure / t, 1.0)  # Normalized participation
            G.add_edge(f'var_{i}', f'template_cap_{j}', weight=participation_weight)
    
    # Add participation edges: variation -> demand constraints  
    # Each variation participates in its own demand constraint
    for i in range(n):
        G.add_edge(f'var_{i}', f'demand_{i}', weight=1.0)
    
    # Add resource consumption edges: variation -> template resources
    for i in range(n):
        for j in range(t):
            demand = d[i] if i < len(d) else 0
            # Resource consumption based on potential slots needed
            consumption = min(demand / (S * total_slots), 1.0) if S > 0 and total_slots > 0 else 0.3
            G.add_edge(f'var_{i}', f'template_res_{j}', weight=consumption)
    
    # Add conflict edges between high-demand variations
    # Large variations compete for limited template space
    if n > 1 and d:
        # Sort variations by demand
        sorted_variations = [(i, d[i]) for i in range(n) if i < len(d)]
        sorted_variations.sort(key=lambda x: x[1], reverse=True)
        
        # Add conflicts between top demanding variations
        high_demand_threshold = max_demand * 0.7 if max_demand > 0 else 0
        high_demand_vars = [i for i, demand in sorted_variations if demand >= high_demand_threshold]
        
        for idx1 in range(len(high_demand_vars)):
            for idx2 in range(idx1 + 1, min(len(high_demand_vars), idx1 + 4)):  # Limit conflicts
                i1, i2 = high_demand_vars[idx1], high_demand_vars[idx2]
                demand1 = d[i1] if i1 < len(d) else 0
                demand2 = d[i2] if i2 < len(d) else 0
                
                # Conflict strength based on combined demand pressure
                combined_pressure = (demand1 + demand2) / total_slots if total_slots > 0 else 0.5
                conflict_weight = min(combined_pressure, 1.0)
                
                if conflict_weight > 0.1:  # Only add meaningful conflicts
                    G.add_edge(f'var_{i1}', f'var_{i2}', weight=conflict_weight)
    
    # Add global complexity node for overall problem difficulty
    if n > 5 and t > 1:  # Only for non-trivial instances
        complexity_weight = min(math.log(n * t) / math.log(50), 1.0)  # Log scaling
        G.add_node('complexity', type=1, weight=complexity_weight)
        
        # Connect to most demanding variations and tight templates
        for i in range(min(n, 5)):  # Top 5 variations
            if i < len(d):
                demand_ratio = d[i] / max_demand if max_demand > 0 else 0.5
                if demand_ratio > 0.5:  # High-demand variations
                    G.add_edge(f'var_{i}', 'complexity', weight=demand_ratio)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()