#!/usr/bin/env python3
"""
Graph converter for Tents problem.
Created using subagent_prompt.md version: v_02

This problem is about placing tents on a grid where each tree must have exactly one adjacent tent.
Key challenges: tree-tent pairing constraints, row/column sum constraints, no adjacent tents constraint.
The spatial arrangement of trees and constraint tightness determine solving difficulty.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the tents problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with cells as variables and constraints as explicit nodes
    - Cells are type 0 (decision variables for tent placement)
    - Trees create adjacency constraints (type 1) 
    - Row/column sum constraints (type 1)
    - No-adjacent-tents constraints for overlapping 2x2 areas (type 1)
    - Weight cells by strategic importance and constraints by tightness
    """
    n = json_data.get('n', 0)
    row_sums = json_data.get('row_sums', [])
    col_sums = json_data.get('col_sums', [])
    a_flat = json_data.get('a', [])
    
    # Convert flat array to 2D grid (row-major order)
    a = []
    for r in range(n):
        row = []
        for c in range(n):
            idx = r * n + c
            row.append(a_flat[idx] if idx < len(a_flat) else 0)
        a.append(row)
    
    G = nx.Graph()
    
    # Calculate problem statistics for weight normalization
    total_tents_needed = sum(row_sums) if row_sums else 0
    total_trees = sum(sum(row) for row in a)
    max_sum = max(max(row_sums) if row_sums else 0, max(col_sums) if col_sums else 0, 1)
    
    # Add cell nodes (type 0) - decision variables for tent placement
    for r in range(n):
        for c in range(n):
            if a[r][c] == 1:  # Tree cell - no decision needed
                continue
                
            # Weight cells by strategic importance
            # Factors: proximity to trees, constraint participation, centrality
            weight = 0.5  # Base weight
            
            # Increase weight for cells adjacent to more trees (more constrained)
            adjacent_trees = 0
            for dr, dc in [(-1,0), (1,0), (0,-1), (0,1)]:
                nr, nc = r + dr, c + dc
                if 0 <= nr < n and 0 <= nc < n and a[nr][nc] == 1:
                    adjacent_trees += 1
            
            # Adjacent to trees = more strategic importance
            if adjacent_trees > 0:
                weight += 0.3 * (adjacent_trees / 4.0)
            
            # Central positions are more constrained by no-adjacent rules
            centrality = 1.0 - (abs(r - n//2) + abs(c - n//2)) / n
            weight += 0.2 * centrality
            
            # Cells in high-demand rows/columns are more important
            row_demand = (row_sums[r] / max_sum) if r < len(row_sums) and max_sum > 0 else 0
            col_demand = (col_sums[c] / max_sum) if c < len(col_sums) and max_sum > 0 else 0
            weight += 0.2 * max(row_demand, col_demand)
            
            G.add_node(f'cell_{r}_{c}', type=0, weight=min(weight, 1.0))
    
    # Add tree adjacency constraint nodes (type 1)
    for r in range(n):
        for c in range(n):
            if a[r][c] == 1:  # This is a tree
                # Each tree must have exactly one adjacent tent
                adjacent_cells = []
                for dr, dc in [(-1,0), (1,0), (0,-1), (0,1)]:
                    nr, nc = r + dr, c + dc
                    if 0 <= nr < n and 0 <= nc < n and a[nr][nc] == 0:
                        adjacent_cells.append((nr, nc))
                
                if adjacent_cells:
                    # Weight by constraint tightness (fewer options = tighter)
                    constraint_weight = 1.0 - (len(adjacent_cells) - 1) / 3.0  # Max 4 adjacent cells
                    constraint_weight = max(constraint_weight, 0.2)
                    
                    constraint_id = f'tree_adj_{r}_{c}'
                    G.add_node(constraint_id, type=1, weight=constraint_weight)
                    
                    # Connect to all adjacent cells
                    for nr, nc in adjacent_cells:
                        # Edge weight reflects importance of this cell for this tree
                        edge_weight = 1.0 / len(adjacent_cells)  # Equal importance for now
                        G.add_edge(f'cell_{nr}_{nc}', constraint_id, weight=edge_weight)
    
    # Add row sum constraint nodes (type 1)
    for r in range(n):
        if r < len(row_sums):
            required_tents = row_sums[r]
            available_cells = sum(1 for c in range(n) if a[r][c] == 0)
            
            if available_cells > 0 and required_tents > 0:
                # Tightness: how constrained is this row?
                tightness = required_tents / available_cells
                # Use non-linear scaling for tightness
                constraint_weight = min(math.pow(tightness, 0.7), 1.0)
                
                constraint_id = f'row_sum_{r}'
                G.add_node(constraint_id, type=1, weight=constraint_weight)
                
                # Connect to all cells in this row
                for c in range(n):
                    if a[r][c] == 0:  # Not a tree
                        # Weight by how much this cell contributes to constraint
                        participation_weight = 1.0 / available_cells
                        G.add_edge(f'cell_{r}_{c}', constraint_id, weight=participation_weight)
    
    # Add column sum constraint nodes (type 1)
    for c in range(n):
        if c < len(col_sums):
            required_tents = col_sums[c]
            available_cells = sum(1 for r in range(n) if a[r][c] == 0)
            
            if available_cells > 0 and required_tents > 0:
                # Tightness: how constrained is this column?
                tightness = required_tents / available_cells
                # Use non-linear scaling for tightness
                constraint_weight = min(math.pow(tightness, 0.7), 1.0)
                
                constraint_id = f'col_sum_{c}'
                G.add_node(constraint_id, type=1, weight=constraint_weight)
                
                # Connect to all cells in this column
                for r in range(n):
                    if a[r][c] == 0:  # Not a tree
                        # Weight by how much this cell contributes to constraint
                        participation_weight = 1.0 / available_cells
                        G.add_edge(f'cell_{r}_{c}', constraint_id, weight=participation_weight)
    
    # Add no-adjacent-tents constraint nodes for overlapping 2x2 areas (type 1)
    for r in range(n-1):
        for c in range(n-1):
            # Check 2x2 area starting at (r,c)
            cells_in_area = []
            for dr in range(2):
                for dc in range(2):
                    nr, nc = r + dr, c + dc
                    if a[nr][nc] == 0:  # Not a tree
                        cells_in_area.append((nr, nc))
            
            if len(cells_in_area) >= 2:  # At least 2 cells that could have tents
                # Weight by potential conflicts (more cells = more complex constraint)
                conflict_potential = len(cells_in_area) / 4.0  # Max 4 cells in 2x2
                constraint_weight = 0.3 + 0.5 * conflict_potential
                
                constraint_id = f'no_adj_{r}_{c}'
                G.add_node(constraint_id, type=1, weight=constraint_weight)
                
                # Connect to all cells in this 2x2 area
                for nr, nc in cells_in_area:
                    # Each cell equally participates in this constraint
                    participation_weight = 1.0 / len(cells_in_area)
                    G.add_edge(f'cell_{nr}_{nc}', constraint_id, weight=participation_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()