#!/usr/bin/env python3
"""
Graph converter for triangular problem.
Created using subagent_prompt.md version: v_02

This problem is about placing 'hearts' on an equilateral triangular grid
such that no three hearts form an equilateral triangle of any size or orientation.
Key challenges: Complex geometric constraints, exponentially many triangle constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the triangular heart placement problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with position nodes and triangle constraint nodes
    - Position nodes (type 0): Each valid grid position, weighted by constraint involvement
    - Triangle constraint nodes (type 1): Each potential equilateral triangle, weighted by size/tightness
    - Edges connect positions to triangles they participate in
    """
    n = json_data.get('n', 0)
    
    if n <= 0:
        return nx.Graph()
    
    G = nx.Graph()
    
    # Valid positions in triangular grid: (i,j) where 1 <= i <= n and 1 <= j <= i
    valid_positions = [(i, j) for i in range(1, n + 1) for j in range(1, i + 1)]
    
    # Add position nodes (type 0) with constraint involvement weights
    position_constraints = {}  # Count how many triangles each position is involved in
    
    # First pass: count constraint involvement for each position
    for i in range(1, n + 1):
        for j in range(1, i + 1):
            position_constraints[(i, j)] = 0
    
    # Count triangles from the constraint: forall(i in 1..n, j in 1..i, k in 1..n-i, m in 0..k-1)
    triangle_constraints = []
    for i in range(1, n + 1):
        for j in range(1, i + 1):
            for k in range(1, n - i + 1):
                for m in range(0, k):
                    # Triangle vertices: (i+m, j), (i+k, j+m), (i+k-m, j+k-m)
                    v1 = (i + m, j)
                    v2 = (i + k, j + m)
                    v3 = (i + k - m, j + k - m)
                    
                    # Check if all vertices are valid positions
                    if (v1 in valid_positions and v2 in valid_positions and v3 in valid_positions):
                        triangle_constraints.append((v1, v2, v3))
                        position_constraints[v1] += 1
                        position_constraints[v2] += 1
                        position_constraints[v3] += 1
    
    # Add position nodes with weights based on constraint involvement
    max_involvement = max(position_constraints.values()) if position_constraints else 1
    for (i, j) in valid_positions:
        involvement = position_constraints.get((i, j), 0)
        # Higher involvement = more constrained = higher weight
        # Also factor in centrality (distance from edges)
        center_i, center_j = (n + 1) / 2, (i + 1) / 2
        distance_from_center = abs(i - center_i) + abs(j - center_j)
        max_distance = n  # approximate max distance
        centrality = 1.0 - (distance_from_center / max_distance)
        
        # Combine involvement and centrality with non-linear scaling
        involvement_weight = math.sqrt(involvement / max_involvement) if max_involvement > 0 else 0.5
        weight = 0.7 * involvement_weight + 0.3 * centrality
        
        G.add_node(f'pos_{i}_{j}', type=0, weight=min(weight, 1.0))
    
    # Add triangle constraint nodes (type 1) with size-based weights
    for idx, (v1, v2, v3) in enumerate(triangle_constraints):
        # Calculate triangle size based on coordinates
        # Larger triangles (k value) are generally easier to satisfy
        i1, j1 = v1
        i2, j2 = v2
        i3, j3 = v3
        
        # Triangle size indicator - larger triangles have lower weight (easier constraint)
        # Use distance between vertices as size measure
        dist1 = abs(i2 - i1) + abs(j2 - j1)
        dist2 = abs(i3 - i1) + abs(j3 - j1)
        dist3 = abs(i3 - i2) + abs(j3 - j2)
        avg_dist = (dist1 + dist2 + dist3) / 3.0
        max_possible_dist = n  # approximate maximum
        
        # Smaller triangles are tighter constraints (higher weight)
        # Use exponential decay for size effect
        size_factor = math.exp(-2.0 * avg_dist / max_possible_dist)
        
        # Edge triangles may be more critical
        edge_bonus = 0.0
        for vi, vj in [v1, v2, v3]:
            if vi == 1 or vi == n or vj == 1 or vj == vi:  # near edges
                edge_bonus += 0.1
        
        weight = min(0.5 + size_factor + edge_bonus, 1.0)
        G.add_node(f'triangle_{idx}', type=1, weight=weight)
    
    # Add bipartite edges: positions to triangles they participate in
    for idx, (v1, v2, v3) in enumerate(triangle_constraints):
        triangle_node = f'triangle_{idx}'
        
        # Edge weight based on the position's role in this triangle
        for pos in [v1, v2, v3]:
            pos_node = f'pos_{pos[0]}_{pos[1]}'
            if G.has_node(pos_node):
                # Weight based on how critical this position is for this constraint
                involvement = position_constraints.get(pos, 1)
                # Normalize and add some variance
                edge_weight = min(0.3 + 0.7 * (involvement / max_involvement), 1.0)
                G.add_edge(pos_node, triangle_node, weight=edge_weight)
    
    # Add conflict edges between positions that frequently appear in triangles together
    # This captures the geometric structure of the problem
    position_cooccurrence = {}
    for v1, v2, v3 in triangle_constraints:
        for pos1 in [v1, v2, v3]:
            for pos2 in [v1, v2, v3]:
                if pos1 < pos2:  # avoid duplicates
                    key = (pos1, pos2)
                    position_cooccurrence[key] = position_cooccurrence.get(key, 0) + 1
    
    # Add conflict edges for highly co-occurring positions
    if position_cooccurrence:
        max_cooccurrence = max(position_cooccurrence.values())
        threshold = max_cooccurrence * 0.7  # Only add edges for high co-occurrence
        
        for (pos1, pos2), count in position_cooccurrence.items():
            if count >= threshold:
                pos1_node = f'pos_{pos1[0]}_{pos1[1]}'
                pos2_node = f'pos_{pos2[0]}_{pos2[1]}'
                if G.has_node(pos1_node) and G.has_node(pos2_node):
                    # Weight based on how often they conflict
                    conflict_strength = count / max_cooccurrence
                    G.add_edge(pos1_node, pos2_node, weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()