#!/usr/bin/env python3
"""
Graph converter for Triangular Hearts problem.
Created using subagent_prompt.md version: v_02

This problem is about placing 'hearts' on an equilateral triangular grid 
such that no three hearts form the corners of an equilateral triangle.
Key challenges: Complex geometric constraints, exponential constraint growth with grid size.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the triangular hearts problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with grid positions and triangle constraints.
    - Grid positions as variables (type 0) with position-based weights
    - Triangle constraints as constraint nodes (type 1) weighted by scope/difficulty
    - Edges connect positions to triangles they participate in
    """
    n = json_data.get('n', 8)
    
    G = nx.Graph()
    
    # Calculate total valid positions in triangular grid
    total_positions = sum(i for i in range(1, n+1))
    
    # Add variable nodes for each valid grid position [i,j] where j <= i
    for i in range(1, n+1):
        for j in range(1, i+1):
            # Position weights: corners and edges are more constrained
            # Distance from center of triangle affects constraint density
            center_row = (n + 1) / 2
            center_col = i / 2
            
            # Calculate centrality - positions closer to center are more constrained
            row_centrality = 1.0 - abs(i - center_row) / n
            col_centrality = 1.0 - abs(j - center_col) / max(i/2, 1)
            
            # Corner positions (j=1 or j=i) have additional constraints
            is_edge = (j == 1) or (j == i)
            edge_penalty = 0.8 if is_edge else 1.0
            
            # Combine factors for final weight
            weight = (row_centrality + col_centrality) / 2 * edge_penalty
            
            G.add_node(f'pos_{i}_{j}', type=0, weight=weight)
    
    # Add constraint nodes for triangle constraints
    # Each constraint prevents 3 positions from forming an equilateral triangle
    constraint_id = 0
    
    # Iterate through all possible triangles as in the MiniZinc model
    for i in range(1, n+1):
        for j in range(1, i+1):
            for k in range(1, n-i+1):
                for m in range(0, k):
                    # The three positions that form a triangle
                    pos1 = (i+m, j)
                    pos2 = (i+k, j+m) 
                    pos3 = (i+k-m, j+k-m)
                    
                    # Check if all positions are valid
                    valid_triangle = True
                    positions = []
                    
                    for row, col in [pos1, pos2, pos3]:
                        if row <= n and col <= row and col >= 1:
                            positions.append((row, col))
                        else:
                            valid_triangle = False
                            break
                    
                    if valid_triangle and len(positions) == 3:
                        # Calculate constraint weight based on triangle size and position
                        triangle_size = k  # Size of the triangle
                        max_size = n - i
                        
                        # Larger triangles are generally easier to satisfy (more spread out)
                        # Smaller triangles are tighter constraints
                        size_factor = math.exp(-2.0 * triangle_size / max(max_size, 1))
                        
                        # Position factor: triangles closer to center are more constrained
                        avg_row = sum(pos[0] for pos in positions) / 3
                        avg_col = sum(pos[1] for pos in positions) / 3
                        center_distance = abs(avg_row - center_row) + abs(avg_col - center_row/2)
                        position_factor = math.exp(-center_distance / n)
                        
                        # Combine factors for constraint weight
                        constraint_weight = min((size_factor + position_factor) / 2, 1.0)
                        
                        constraint_node = f'triangle_{constraint_id}'
                        G.add_node(constraint_node, type=1, weight=constraint_weight)
                        
                        # Add edges from constraint to participating positions
                        for row, col in positions:
                            pos_node = f'pos_{row}_{col}'
                            if G.has_node(pos_node):
                                # Edge weight reflects how critical this position is for this constraint
                                # Positions that are in more constraints get higher edge weights
                                G.add_edge(pos_node, constraint_node, weight=0.8)
                        
                        constraint_id += 1
    
    # Add a global objective constraint node representing the maximization goal
    # This connects to all positions with weights based on their strategic value
    G.add_node('objective_constraint', type=1, weight=1.0)
    
    for i in range(1, n+1):
        for j in range(1, i+1):
            pos_node = f'pos_{i}_{j}'
            # Strategic value: positions that are less constrained are more valuable
            # for maximizing the objective
            pos_degree = G.degree(pos_node)
            max_degree = max(G.degree(node) for node in G.nodes() if 'pos_' in node) if G.nodes() else 1
            
            strategic_value = 1.0 - (pos_degree / max(max_degree, 1))
            G.add_edge(pos_node, 'objective_constraint', weight=strategic_value)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()