#!/usr/bin/env python3
"""
Graph converter for Trucking problem.
Created using subagent_prompt.md version: v_02

This problem is about truck scheduling with cost minimization.
Key challenges: Meeting demand with minimum cost while respecting special constraints on certain trucks.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the trucking problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with decision variables, constraints, and resources
    - Decision variables: truck-time slot combinations (x[i,t])
    - Demand constraints: Each time period must meet demand
    - Special constraints: Truck1 and Truck2 availability restrictions
    - Resources: Individual trucks and time periods
    - Weights reflect cost efficiency, demand pressure, and constraint tightness
    """
    T = json_data.get('T', 0)  # time periods
    N = json_data.get('N', 0)  # trucks
    truck1 = json_data.get('Truck1', 1) - 1  # Convert to 0-indexed
    truck2 = json_data.get('Truck2', 1) - 1  # Convert to 0-indexed
    costs = json_data.get('Cost', [])
    loads = json_data.get('Loads', [])
    demands = json_data.get('Demand', [])
    
    G = nx.Graph()
    
    # Calculate normalization factors
    max_cost = max(costs) if costs else 1
    min_cost = min(costs) if costs else 1
    max_load = max(loads) if loads else 1
    max_demand = max(demands) if demands else 1
    total_capacity = sum(loads) if loads else 1
    
    # Type 0: Decision variable nodes (truck-time combinations)
    for i in range(N):
        for t in range(T):
            if i < len(costs) and i < len(loads):
                # Cost efficiency: lower cost per unit load is better
                cost_per_load = costs[i] / max(loads[i], 1)
                efficiency = 1.0 - (cost_per_load / (max_cost / 1))  # Normalize to [0,1]
                
                # Demand pressure: higher demand periods make decisions more critical
                demand_pressure = demands[t] / max_demand if t < len(demands) else 0.5
                
                # Combine efficiency and demand pressure
                weight = (efficiency + demand_pressure) / 2
                weight = max(0.1, min(1.0, weight))  # Clamp to [0.1, 1.0]
                
                G.add_node(f'truck_{i}_time_{t}', type=0, weight=weight)
    
    # Type 1: Demand constraint nodes
    for t in range(T):
        if t < len(demands):
            # Constraint tightness: how much total capacity vs demand
            demand = demands[t]
            tightness = min(1.0, demand / total_capacity)
            # High demand periods are tighter constraints
            constraint_weight = 0.3 + 0.7 * tightness
            G.add_node(f'demand_constraint_{t}', type=1, weight=constraint_weight)
    
    # Type 1: Special truck constraint nodes
    # Truck1 constraint (3-period windows)
    for tau in range(T-2):
        # Window tightness depends on demand in the window
        window_demand = sum(demands[tau:tau+3]) if tau+2 < len(demands) else 0
        window_tightness = min(1.0, window_demand / (3 * max_demand))
        weight = 0.5 + 0.5 * window_tightness
        G.add_node(f'truck1_constraint_{tau}', type=1, weight=weight)
    
    # Truck2 constraint (2-period windows)  
    for tau in range(T-1):
        # Window tightness depends on demand in the window
        window_demand = sum(demands[tau:tau+2]) if tau+1 < len(demands) else 0
        window_tightness = min(1.0, window_demand / (2 * max_demand))
        weight = 0.5 + 0.5 * window_tightness
        G.add_node(f'truck2_constraint_{tau}', type=1, weight=weight)
    
    # Type 2: Resource nodes (trucks)
    for i in range(N):
        if i < len(costs) and i < len(loads):
            # Truck value: combination of capacity and cost efficiency
            capacity_ratio = loads[i] / max_load
            cost_ratio = 1.0 - (costs[i] - min_cost) / max(max_cost - min_cost, 1)
            truck_value = (capacity_ratio + cost_ratio) / 2
            G.add_node(f'truck_resource_{i}', type=2, weight=truck_value)
    
    # Type 2: Time period resource nodes
    for t in range(T):
        if t < len(demands):
            # Time period criticality based on demand
            demand_ratio = demands[t] / max_demand
            # Use non-linear scaling for demand pressure
            criticality = math.pow(demand_ratio, 1.5)  # Exponential scaling
            G.add_node(f'time_resource_{t}', type=2, weight=criticality)
    
    # Bipartite edges: Decision variables to demand constraints
    for t in range(T):
        demand = demands[t] if t < len(demands) else 0
        for i in range(N):
            if i < len(loads):
                # Edge weight based on how much this truck contributes to demand satisfaction
                contribution = min(1.0, loads[i] / max(demand, 1))
                G.add_edge(f'truck_{i}_time_{t}', f'demand_constraint_{t}', 
                          weight=contribution)
    
    # Bipartite edges: Decision variables to special constraints
    # Truck1 constraints
    if truck1 < N:
        for tau in range(T-2):
            for t in range(tau, min(tau+3, T)):
                # All truck1 decisions in this window connect to constraint
                G.add_edge(f'truck_{truck1}_time_{t}', f'truck1_constraint_{tau}', 
                          weight=1.0)
    
    # Truck2 constraints
    if truck2 < N:
        for tau in range(T-1):
            for t in range(tau, min(tau+2, T)):
                # All truck2 decisions in this window connect to constraint
                G.add_edge(f'truck_{truck2}_time_{t}', f'truck2_constraint_{tau}', 
                          weight=1.0)
    
    # Resource usage edges: Decision variables to truck resources
    for i in range(N):
        for t in range(T):
            # Cost-based edge weight with exponential scaling
            if i < len(costs):
                cost_factor = costs[i] / max_cost
                # Use exponential decay for cost sensitivity
                weight = math.exp(-2.0 * cost_factor)
                G.add_edge(f'truck_{i}_time_{t}', f'truck_resource_{i}', 
                          weight=weight)
    
    # Temporal edges: Decision variables to time resources
    for t in range(T):
        for i in range(N):
            # Time pressure based on demand and available capacity
            if t < len(demands) and i < len(loads):
                demand_pressure = demands[t] / max_demand
                capacity_contribution = loads[i] / total_capacity
                weight = (demand_pressure + capacity_contribution) / 2
                G.add_edge(f'truck_{i}_time_{t}', f'time_resource_{t}', 
                          weight=weight)
    
    # Conflict edges: High-cost trucks competing for high-demand periods
    high_demand_periods = [t for t in range(T) 
                          if t < len(demands) and demands[t] > 0.8 * max_demand]
    high_cost_trucks = [i for i in range(N) 
                       if i < len(costs) and costs[i] > 0.8 * max_cost]
    
    for t in high_demand_periods:
        for i1 in high_cost_trucks:
            for i2 in high_cost_trucks:
                if i1 < i2:  # Avoid duplicate edges
                    # Conflict weight based on combined cost inefficiency
                    cost_sum = (costs[i1] + costs[i2]) / (2 * max_cost)
                    G.add_edge(f'truck_{i1}_time_{t}', f'truck_{i2}_time_{t}', 
                              weight=cost_sum)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()