#!/usr/bin/env python3
"""
Graph converter for trucking_hl problem.
Created using subagent_prompt.md version: v_02

This problem is about truck scheduling across multiple periods.
Each truck has cost, capacity, and availability limits.
Key challenges: satisfying demands while minimizing costs and respecting truck limits.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the trucking_hl problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with explicit constraint nodes
    - Truck nodes (type 0): Decision variables for each truck-period combination
    - Period demand constraint nodes (type 1): Ensure each period's demand is met
    - Truck limit constraint nodes (type 1): Enforce truck availability windows
    - Period nodes (type 2): Represent time periods as resources
    - Truck resource nodes (type 2): Represent truck resources
    
    What makes instances hard:
    - High demand relative to available capacity
    - Tight truck availability limits
    - Poor cost-efficiency ratio of trucks
    """
    # Access data from json_data dict
    P = json_data.get('P', 0)  # periods
    T = json_data.get('T', 0)  # trucks
    Demand = json_data.get('Demand', [])
    Cost = json_data.get('Cost', [])
    Loads = json_data.get('Loads', [])
    Limit = json_data.get('Limit', [])
    
    G = nx.Graph()
    
    # Calculate metrics for weighting
    max_cost = max(Cost) if Cost else 1
    max_load = max(Loads) if Loads else 1
    max_demand = max(Demand) if Demand else 1
    max_limit = max(Limit) if Limit else 1
    total_demand = sum(Demand)
    total_capacity = sum(Loads) * P  # theoretical max if all trucks used all periods
    
    # Truck-period decision variable nodes (type 0)
    for t in range(T):
        for p in range(P):
            # Weight by cost-efficiency: higher cost = higher weight (harder decision)
            cost_ratio = Cost[t] / max_cost if t < len(Cost) else 0.5
            load_ratio = Loads[t] / max_load if t < len(Loads) else 0.5
            # Inefficient trucks (high cost, low load) are harder decisions
            efficiency = cost_ratio / load_ratio if load_ratio > 0 else 1.0
            weight = min(efficiency / 2.0, 1.0)  # normalize to [0,1]
            G.add_node(f'truck_{t}_period_{p}', type=0, weight=weight)
    
    # Period demand constraint nodes (type 1)
    for p in range(P):
        demand = Demand[p] if p < len(Demand) else 0
        # Weight by demand pressure: higher demand = tighter constraint
        demand_pressure = demand / max_demand if max_demand > 0 else 0.5
        # Also consider if demand is hard to satisfy with available trucks
        period_capacity = sum(Loads)  # max capacity available in this period
        tightness = demand / period_capacity if period_capacity > 0 else 1.0
        weight = min((demand_pressure + tightness) / 2.0, 1.0)
        G.add_node(f'demand_constraint_{p}', type=1, weight=weight)
    
    # Truck limit constraint nodes (type 1) - for trucks with limits > 0
    for t in range(T):
        limit = Limit[t] if t < len(Limit) else 0
        if limit > 0:  # Only create constraint nodes for trucks with actual limits
            # Weight by restrictiveness: smaller limit relative to periods = tighter
            restrictiveness = 1.0 - (limit / P) if P > 0 else 0.5
            G.add_node(f'truck_limit_{t}', type=1, weight=restrictiveness)
    
    # Period resource nodes (type 2)
    for p in range(P):
        demand = Demand[p] if p < len(Demand) else 0
        # Weight by resource scarcity
        scarcity = demand / max_demand if max_demand > 0 else 0.5
        G.add_node(f'period_{p}', type=2, weight=scarcity)
    
    # Truck resource nodes (type 2)
    for t in range(T):
        cost = Cost[t] if t < len(Cost) else 0
        load = Loads[t] if t < len(Loads) else 0
        # Weight by resource quality (lower cost, higher load = better resource)
        cost_norm = cost / max_cost if max_cost > 0 else 0.5
        load_norm = load / max_load if max_load > 0 else 0.5
        quality = (1.0 - cost_norm + load_norm) / 2.0  # better trucks have lower weight
        G.add_node(f'truck_resource_{t}', type=2, weight=1.0 - quality)
    
    # Bipartite edges: truck-period variables to constraints they participate in
    
    # Edges from truck-period variables to demand constraints
    for p in range(P):
        demand = Demand[p] if p < len(Demand) else 0
        for t in range(T):
            load = Loads[t] if t < len(Loads) else 0
            # Edge weight represents contribution to satisfying demand
            contribution = load / demand if demand > 0 else 0.5
            weight = min(contribution, 1.0)
            G.add_edge(f'truck_{t}_period_{p}', f'demand_constraint_{p}', weight=weight)
    
    # Edges from truck-period variables to truck limit constraints
    for t in range(T):
        limit = Limit[t] if t < len(Limit) else 0
        if limit > 0:  # Only for trucks with actual limits
            for p in range(P):
                # Edge weight represents impact on truck's availability window
                # Trucks with tighter limits have higher edge weights
                impact = 1.0 / limit if limit > 0 else 1.0
                weight = min(impact, 1.0)
                G.add_edge(f'truck_{t}_period_{p}', f'truck_limit_{t}', weight=weight)
    
    # Edges from truck-period variables to period resources
    for t in range(T):
        for p in range(P):
            load = Loads[t] if t < len(Loads) else 0
            # Edge weight represents resource consumption
            consumption = load / max_load if max_load > 0 else 0.5
            G.add_edge(f'truck_{t}_period_{p}', f'period_{p}', weight=consumption)
    
    # Edges from truck-period variables to truck resources
    for t in range(T):
        for p in range(P):
            # Edge weight represents usage of truck resource
            # More periods using same truck = higher total weight
            usage = 1.0 / P  # each period uses 1/P of truck's availability
            G.add_edge(f'truck_{t}_period_{p}', f'truck_resource_{t}', weight=usage)
    
    # Add conflict edges between truck-period variables for trucks with limits
    for t in range(T):
        limit = Limit[t] if t < len(Limit) else 0
        if limit > 0 and limit < P:  # Truck has actual availability constraints
            # Add conflicts between periods that are too close given the limit
            for p1 in range(P):
                for p2 in range(p1 + 1, min(p1 + limit, P)):
                    # These periods conflict if truck t is used in both
                    # Weight by how restrictive the conflict is
                    conflict_strength = 1.0 - (abs(p2 - p1) / limit) if limit > 0 else 0.5
                    if conflict_strength > 0.1:  # Only add meaningful conflicts
                        G.add_edge(f'truck_{t}_period_{p1}', f'truck_{t}_period_{p2}', 
                                 weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()