#!/usr/bin/env python3
"""
Graph converter for Traveling Tournament Problem with Predefined Venues (TTPPV).
Created using subagent_prompt.md version: v_02

This problem is about scheduling round-robin tournaments where venue assignments 
are predefined, minimizing total travel distance while respecting pattern constraints.
Key challenges: venue assignment optimization, travel cost minimization, 
consecutive home/away game limits, and circular distance structure.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the TTPPV problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph that captures scheduling constraints,
    venue assignments, and travel relationships.
    - Team nodes (type 0): Decision makers that need scheduling
    - Constraint nodes (type 1): Various scheduling constraints 
    - Round nodes (type 2): Time slot resources
    - Edges represent participation in constraints and resource usage
    
    Key entities: Teams, rounds, venue assignments, travel patterns
    What makes instances hard: Imbalanced venue assignments, conflicting 
    travel requirements, tight pattern constraints
    """
    # Access data from json_data dict
    nb_teams = json_data.get('nbTeams', 0)
    pv_flat = json_data.get('pv', [])
    
    if nb_teams == 0:
        # Return minimal graph for empty instances
        G = nx.Graph()
        G.add_node('dummy', type=0, weight=0.5)
        return G
    
    nb_rounds = nb_teams - 1
    
    # Reconstruct pv matrix from flat array (row-major order)
    pv = {}
    for i in range(nb_teams):
        for j in range(nb_teams):
            if i != j:  # No self-games
                idx = i * nb_teams + j
                if idx < len(pv_flat):
                    pv[(i+1, j+1)] = pv_flat[idx]  # 1-indexed teams
    
    # Calculate circular distances
    def circular_distance(i, j, n):
        """Calculate circular distance between teams i and j."""
        if i == j:
            return 0
        diff = abs(i - j)
        return min(diff, n - diff)
    
    # Create graph
    G = nx.Graph()
    
    # Team nodes (type 0) - weighted by scheduling difficulty
    for i in range(1, nb_teams + 1):
        # Calculate venue balance - teams with more away games are harder to schedule
        away_games = sum(1 for j in range(1, nb_teams + 1) 
                        if i != j and pv.get((i, j), 2) == 2)
        venue_imbalance = abs(away_games - nb_rounds / 2) / (nb_rounds / 2)
        
        # Calculate average travel burden based on predefined venues
        total_travel = 0
        travel_count = 0
        for j in range(1, nb_teams + 1):
            if i != j:
                if pv.get((i, j), 2) == 2:  # Away game
                    total_travel += circular_distance(i, j, nb_teams)
                    travel_count += 1
        
        avg_travel = total_travel / max(travel_count, 1)
        travel_burden = avg_travel / max(nb_teams // 2, 1)
        
        # Combined difficulty: venue imbalance + travel burden
        difficulty = (venue_imbalance + travel_burden) / 2
        
        G.add_node(f'team_{i}', type=0, weight=min(difficulty, 1.0))
    
    # Round nodes (type 2) - time slot resources
    for r in range(1, nb_rounds + 1):
        # Later rounds are more constrained due to accumulated patterns
        constraint_pressure = r / nb_rounds
        G.add_node(f'round_{r}', type=2, weight=constraint_pressure)
    
    # Constraint nodes (type 1) for different constraint types
    
    # 1. Opponent uniqueness constraints (one per team)
    for i in range(1, nb_teams + 1):
        # All-different constraint on opponents for team i
        G.add_node(f'alldiff_opp_team_{i}', type=1, weight=0.8)
    
    # 2. Round pairing constraints (one per round)
    for r in range(1, nb_rounds + 1):
        # All-different constraint on opponents in round r
        G.add_node(f'alldiff_round_{r}', type=1, weight=0.7)
    
    # 3. Venue assignment constraints (one per team-opponent pair)
    venue_constraint_weights = []
    for i in range(1, nb_teams + 1):
        for j in range(1, nb_teams + 1):
            if i != j:
                venue = pv.get((i, j), 2)
                # Constraint linking venue to opponent choice
                constraint_id = f'venue_{i}_vs_{j}'
                # Weight by travel distance if away game
                if venue == 2:  # Away game
                    travel_dist = circular_distance(i, j, nb_teams)
                    weight = travel_dist / max(nb_teams // 2, 1)
                else:  # Home game
                    weight = 0.3  # Lower weight for home games
                
                venue_constraint_weights.append(weight)
                G.add_node(constraint_id, type=1, weight=min(weight, 1.0))
    
    # 4. Pattern constraints (consecutive home/away limits)
    for i in range(1, nb_teams + 1):
        # Regular expression constraint for max 3 consecutive home/away
        # This is a complex global constraint
        G.add_node(f'pattern_team_{i}', type=1, weight=0.9)
    
    # 5. Travel optimization constraint nodes
    # Create constraints for high-travel team pairs
    high_travel_pairs = []
    for i in range(1, nb_teams + 1):
        for j in range(1, nb_teams + 1):
            if i != j and pv.get((i, j), 2) == 2:  # Away game
                travel_dist = circular_distance(i, j, nb_teams)
                if travel_dist >= nb_teams // 3:  # High travel distance
                    high_travel_pairs.append((i, j, travel_dist))
    
    for i, j, dist in high_travel_pairs:
        # Constraint representing travel cost optimization
        weight = dist / max(nb_teams // 2, 1)
        G.add_node(f'travel_{i}_to_{j}', type=1, weight=min(weight, 1.0))
    
    # Add edges: team-constraint participation
    
    # Teams participate in their opponent uniqueness constraints
    for i in range(1, nb_teams + 1):
        G.add_edge(f'team_{i}', f'alldiff_opp_team_{i}', weight=1.0)
    
    # Teams participate in pattern constraints
    for i in range(1, nb_teams + 1):
        # Pattern constraint weight based on venue imbalance
        away_games = sum(1 for j in range(1, nb_teams + 1) 
                        if i != j and pv.get((i, j), 2) == 2)
        pattern_difficulty = abs(away_games - nb_rounds / 2) / nb_rounds
        G.add_edge(f'team_{i}', f'pattern_team_{i}', 
                  weight=min(pattern_difficulty * 2, 1.0))
    
    # Teams participate in venue assignment constraints
    for i in range(1, nb_teams + 1):
        for j in range(1, nb_teams + 1):
            if i != j:
                constraint_id = f'venue_{i}_vs_{j}'
                venue = pv.get((i, j), 2)
                if venue == 2:  # Away game - higher participation weight
                    weight = 0.8
                else:  # Home game
                    weight = 0.4
                G.add_edge(f'team_{i}', constraint_id, weight=weight)
    
    # Teams participate in travel constraints
    for i, j, dist in high_travel_pairs:
        travel_weight = dist / max(nb_teams // 2, 1)
        G.add_edge(f'team_{i}', f'travel_{i}_to_{j}', 
                  weight=min(travel_weight, 1.0))
    
    # Teams use round resources
    for i in range(1, nb_teams + 1):
        for r in range(1, nb_rounds + 1):
            # Each team must be scheduled in each round
            # Weight by how constrained the team is in that round
            round_weight = r / nb_rounds  # Later rounds more constrained
            G.add_edge(f'team_{i}', f'round_{r}', weight=round_weight)
    
    # Round constraints participate with rounds
    for r in range(1, nb_rounds + 1):
        G.add_edge(f'alldiff_round_{r}', f'round_{r}', weight=0.9)
    
    # Add conflict edges between teams with incompatible travel patterns
    for i in range(1, nb_teams + 1):
        for j in range(i + 1, nb_teams + 1):
            # Check if these teams have conflicting high-travel requirements
            i_high_travel = sum(1 for k in range(1, nb_teams + 1) 
                              if k != i and pv.get((i, k), 2) == 2 and 
                              circular_distance(i, k, nb_teams) >= nb_teams // 3)
            j_high_travel = sum(1 for k in range(1, nb_teams + 1) 
                              if k != j and pv.get((j, k), 2) == 2 and 
                              circular_distance(j, k, nb_teams) >= nb_teams // 3)
            
            if i_high_travel > 0 and j_high_travel > 0:
                # Both teams have high travel - potential scheduling conflict
                conflict_strength = (i_high_travel + j_high_travel) / (2 * nb_rounds)
                if conflict_strength > 0.3:  # Significant conflict
                    G.add_edge(f'team_{i}', f'team_{j}', 
                              weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()