#!/usr/bin/env python3
"""
Graph converter for Traveling Tournament Problem with Predefined Venues (TTPPV).
Created using subagent_prompt.md version: v_02

This problem is about scheduling a round-robin tournament where teams travel between venues,
with predefined home/away assignments, minimizing total travel distance.
Key challenges: scheduling constraints, travel minimization, venue constraints, consecutive games limits.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the TTPPV problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model teams, rounds, and various constraint types as a bipartite graph
    - Teams (type 0): Decision entities with travel burden weights
    - Constraint nodes (type 1): Various constraint types with scope/tightness weights  
    - Edges model participation in constraints and conflicts
    - Focus on travel patterns, venue assignments, and scheduling constraints
    """
    # Access data directly from json_data dict
    nb_teams = json_data.get('nbTeams', 0)
    pv_flat = json_data.get('pv', [])
    
    if nb_teams == 0:
        # Return minimal graph for empty data
        G = nx.Graph()
        G.add_node('dummy', type=0, weight=0.5)
        return G
    
    nb_rounds = nb_teams - 1
    
    # Reconstruct 2D venue matrix from flat array
    pv = []
    for i in range(nb_teams):
        row = []
        for j in range(nb_teams):
            if i * nb_teams + j < len(pv_flat):
                row.append(pv_flat[i * nb_teams + j])
            else:
                row.append(2)  # default away
        pv.append(row)
    
    # Create graph
    G = nx.Graph()
    
    # Calculate team weights based on travel burden and venue imbalance
    max_home_games = max(sum(1 for j in range(nb_teams) if i != j and pv[i][j] == 1) for i in range(nb_teams)) if nb_teams > 1 else 1
    
    # Add team nodes (type 0) with travel burden weights
    for i in range(nb_teams):
        # Count home games for this team (excluding self-play)
        home_games = sum(1 for j in range(nb_teams) if i != j and pv[i][j] == 1)
        away_games = nb_rounds - home_games
        
        # Travel burden: more away games = higher weight (more constrained)
        travel_burden = away_games / nb_rounds if nb_rounds > 0 else 0.5
        
        # Venue imbalance penalty
        ideal_home = nb_rounds / 2
        imbalance = abs(home_games - ideal_home) / ideal_home if ideal_home > 0 else 0
        
        # Combined weight with non-linear scaling
        weight = min(1.0, 0.3 + 0.4 * travel_burden + 0.3 * math.sqrt(imbalance))
        G.add_node(f'team_{i}', type=0, weight=weight)
    
    # Add constraint nodes (type 1)
    
    # 1. Round constraints - each round must have valid pairings
    for k in range(nb_rounds):
        # Round scope affects constraint tightness
        scope_weight = 0.7 + 0.3 * (k + 1) / nb_rounds  # later rounds slightly harder
        G.add_node(f'round_{k}', type=1, weight=scope_weight)
    
    # 2. Opponent uniqueness constraints - each team plays others exactly once
    for i in range(nb_teams):
        # Teams with unbalanced venues are harder to schedule
        home_games = sum(1 for j in range(nb_teams) if i != j and pv[i][j] == 1)
        imbalance = abs(home_games - nb_rounds/2) / (nb_rounds/2) if nb_rounds > 0 else 0
        weight = 0.6 + 0.4 * math.sqrt(imbalance)
        G.add_node(f'unique_opponents_{i}', type=1, weight=weight)
    
    # 3. Symmetry constraints - if i plays j, then j plays i
    constraint_count = 0
    for i in range(nb_teams):
        for j in range(i + 1, nb_teams):
            # Venue conflicts make symmetry harder
            venue_conflict = 0.5
            if pv[i][j] == pv[j][i]:  # both want same venue type
                venue_conflict = 1.0
            G.add_node(f'symmetry_{constraint_count}', type=1, weight=0.5 + 0.5 * venue_conflict)
            constraint_count += 1
    
    # 4. Consecutive games constraints - max 3 consecutive home/away
    for i in range(nb_teams):
        # Count venue changes needed
        team_venues = [pv[i][j] for j in range(nb_teams) if i != j]
        venue_changes = sum(1 for k in range(len(team_venues)-1) if team_venues[k] != team_venues[k+1])
        
        # More venue changes = harder to satisfy consecutive limits
        change_density = venue_changes / max(nb_rounds - 1, 1)
        weight = 0.4 + 0.6 * (1.0 - math.exp(-3.0 * change_density))
        G.add_node(f'consecutive_limit_{i}', type=1, weight=weight)
    
    # 5. Travel distance constraint (global objective)
    # Calculate total minimum possible travel (lower bound)
    min_travel = 0
    for i in range(nb_teams):
        away_games = sum(1 for j in range(nb_teams) if i != j and pv[i][j] == 2)
        # Circular distance: minimum is 1 for adjacent teams
        min_travel += away_games  # rough approximation
    
    # Travel constraint tightness based on venue distribution
    total_possible_travel = nb_teams * nb_rounds * (nb_teams // 2)  # rough upper bound
    travel_tightness = min_travel / max(total_possible_travel, 1)
    G.add_node('travel_minimization', type=1, weight=0.3 + 0.7 * math.sqrt(travel_tightness))
    
    # Add edges modeling constraint participation
    
    # Teams participate in round constraints
    for i in range(nb_teams):
        for k in range(nb_rounds):
            # Participation strength based on team's venue flexibility in this round
            strength = 0.7  # base participation
            G.add_edge(f'team_{i}', f'round_{k}', weight=strength)
    
    # Teams participate in their opponent uniqueness constraints
    for i in range(nb_teams):
        G.add_edge(f'team_{i}', f'unique_opponents_{i}', weight=1.0)
    
    # Teams participate in symmetry constraints
    constraint_count = 0
    for i in range(nb_teams):
        for j in range(i + 1, nb_teams):
            # Both teams participate in their symmetry constraint
            venue_importance = 0.6 + 0.4 * abs(pv[i][j] - pv[j][i])  # higher if venues differ
            G.add_edge(f'team_{i}', f'symmetry_{constraint_count}', weight=venue_importance)
            G.add_edge(f'team_{j}', f'symmetry_{constraint_count}', weight=venue_importance)
            constraint_count += 1
    
    # Teams participate in their consecutive game constraints
    for i in range(nb_teams):
        G.add_edge(f'team_{i}', f'consecutive_limit_{i}', weight=0.9)
    
    # All teams participate in travel minimization
    for i in range(nb_teams):
        away_games = sum(1 for j in range(nb_teams) if i != j and pv[i][j] == 2)
        travel_participation = away_games / max(nb_rounds, 1)  # teams with more away games contribute more
        G.add_edge(f'team_{i}', 'travel_minimization', weight=0.4 + 0.6 * travel_participation)
    
    # Add conflict edges between teams with venue conflicts
    for i in range(nb_teams):
        for j in range(i + 1, nb_teams):
            # Check if these teams have venue conflicts (both want home vs each other)
            if pv[i][j] == 1 and pv[j][i] == 1:  # impossible - both can't play at home
                G.add_edge(f'team_{i}', f'team_{j}', weight=1.0)
            elif pv[i][j] == 2 and pv[j][i] == 2:  # both want away - also conflict
                G.add_edge(f'team_{i}', f'team_{j}', weight=0.8)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()