#!/usr/bin/env python3
"""
Graph converter for Vehicle Routing Problem.
Created using subagent_prompt.md version: v_01

This problem routes K vehicles from a central depot to serve N customers,
minimizing total travel distance while respecting vehicle capacity constraints.
Key challenges: capacity management, route optimization, subtour elimination.
"""

import sys
import json
import networkx as nx
from pathlib import Path
import math


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the VRP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with explicit constraint nodes
    - Customers as type 0 nodes (weighted by demand density)
    - Constraints as type 1 nodes (capacity, degree, subtour elimination)
    - Depot as type 2 node (resource hub)
    - Model capacity tightness, geographic clustering, demand distribution
    """
    
    # Extract data
    N = json_data.get('N', 0)  # Number of customers (depot is node 0, not counted)
    capacity = json_data.get('Capacity', 1)
    demands = json_data.get('Demand', [])
    distance_flat = json_data.get('Distance', [])
    
    # Reconstruct distance matrix from flat array
    # Distance matrix is (N+1) x (N+1) including depot
    total_nodes = N + 1
    distance_matrix = []
    for i in range(total_nodes):
        row = distance_flat[i * total_nodes:(i + 1) * total_nodes]
        distance_matrix.append(row)
    
    G = nx.Graph()
    
    # Calculate metrics for weighting
    total_demand = sum(demands)
    max_demand = max(demands) if demands else 1
    max_distance = max(distance_flat) if distance_flat else 1
    
    # 1. Customer nodes (type 0) - weighted by demand density and centrality
    for i in range(1, N + 1):  # Customers are nodes 1..N
        demand = demands[i - 1] if i - 1 < len(demands) else 0
        
        # Calculate centrality based on distances to all other locations
        avg_distance = sum(distance_matrix[i][j] for j in range(total_nodes)) / total_nodes
        centrality = 1.0 - (avg_distance / max_distance)
        
        # Demand density: higher demand = higher weight
        demand_weight = demand / max_demand if max_demand > 0 else 0.5
        
        # Combined weight emphasizing both demand and centrality
        node_weight = 0.6 * demand_weight + 0.4 * centrality
        
        G.add_node(f'customer_{i}', type=0, weight=node_weight)
    
    # 2. Depot node (type 2) - central resource hub
    depot_centrality = 1.0 - (sum(distance_matrix[0]) / total_nodes / max_distance)
    G.add_node('depot', type=2, weight=depot_centrality)
    
    # 3. Constraint nodes (type 1)
    
    # 3a. Global capacity constraint - measures overall tightness
    capacity_utilization = total_demand / capacity if capacity > 0 else 1.0
    capacity_tightness = min(capacity_utilization, 1.0)
    G.add_node('capacity_constraint', type=1, weight=capacity_tightness)
    
    # 3b. Individual customer degree constraints (indegree = outdegree = 1)
    for i in range(1, N + 1):
        # Weight by demand - high demand customers have tighter constraints
        demand = demands[i - 1] if i - 1 < len(demands) else 0
        constraint_weight = demand / max_demand if max_demand > 0 else 0.5
        G.add_node(f'degree_constraint_{i}', type=1, weight=constraint_weight)
    
    # 3c. Depot degree constraints (limited number of vehicles K)
    # K = N in the model, but effective K is much smaller
    effective_K = min(8, N)  # Reasonable upper bound for vehicles
    depot_constraint_tightness = min(N / effective_K, 1.0)
    G.add_node('depot_degree_constraint', type=1, weight=depot_constraint_tightness)
    
    # 3d. Subtour elimination constraints - one per customer pair
    # Focus on nearby customer pairs that might form subtours
    subtour_constraints = 0
    distance_threshold = max_distance * 0.3  # Consider nearby pairs
    
    for i in range(1, N + 1):
        for j in range(i + 1, N + 1):
            if distance_matrix[i][j] <= distance_threshold:
                # Weight by proximity and demand similarity
                proximity = 1.0 - (distance_matrix[i][j] / distance_threshold)
                demand_i = demands[i - 1] if i - 1 < len(demands) else 0
                demand_j = demands[j - 1] if j - 1 < len(demands) else 0
                demand_similarity = 1.0 - abs(demand_i - demand_j) / max_demand if max_demand > 0 else 0.5
                
                constraint_weight = 0.7 * proximity + 0.3 * demand_similarity
                G.add_node(f'subtour_{i}_{j}', type=1, weight=constraint_weight)
                subtour_constraints += 1
                
                if subtour_constraints >= N:  # Limit to avoid too many nodes
                    break
        if subtour_constraints >= N:
            break
    
    # 4. Edges - bipartite connections and some conflicts
    
    # 4a. Customer-constraint participation edges
    for i in range(1, N + 1):
        customer = f'customer_{i}'
        demand = demands[i - 1] if i - 1 < len(demands) else 0
        
        # Connect to capacity constraint (weighted by demand contribution)
        capacity_contribution = demand / capacity if capacity > 0 else 0.5
        G.add_edge(customer, 'capacity_constraint', weight=min(capacity_contribution, 1.0))
        
        # Connect to own degree constraint
        G.add_edge(customer, f'degree_constraint_{i}', weight=1.0)
        
        # Connect to depot degree constraint (weighted by distance to depot)
        depot_distance = distance_matrix[0][i] if i < len(distance_matrix[0]) else max_distance
        depot_weight = math.exp(-3.0 * depot_distance / max_distance)  # Exponential decay
        G.add_edge(customer, 'depot_degree_constraint', weight=depot_weight)
    
    # 4b. Depot connections
    depot_avg_distance = sum(distance_matrix[0][1:N+1]) / N if N > 0 else max_distance
    depot_connection_weight = 1.0 - (depot_avg_distance / max_distance)
    G.add_edge('depot', 'depot_degree_constraint', weight=depot_connection_weight)
    G.add_edge('depot', 'capacity_constraint', weight=0.8)  # Central to capacity management
    
    # 4c. Subtour constraint edges
    for i in range(1, N + 1):
        for j in range(i + 1, N + 1):
            subtour_node = f'subtour_{i}_{j}'
            if subtour_node in G.nodes:
                # Connect both customers to this subtour constraint
                distance_ij = distance_matrix[i][j] if i < len(distance_matrix) and j < len(distance_matrix[i]) else max_distance
                edge_weight = math.exp(-2.0 * distance_ij / max_distance)
                
                G.add_edge(f'customer_{i}', subtour_node, weight=edge_weight)
                G.add_edge(f'customer_{j}', subtour_node, weight=edge_weight)
    
    # 4d. Customer-customer conflict edges for capacity competition
    # Add edges between customers that together exceed capacity
    for i in range(1, N + 1):
        for j in range(i + 1, N + 1):
            demand_i = demands[i - 1] if i - 1 < len(demands) else 0
            demand_j = demands[j - 1] if j - 1 < len(demands) else 0
            
            # If combined demand exceeds capacity, they conflict
            if demand_i + demand_j > capacity * 0.8:  # 80% threshold for conflict
                conflict_strength = (demand_i + demand_j) / capacity - 0.8
                conflict_weight = min(conflict_strength, 1.0)
                G.add_edge(f'customer_{i}', f'customer_{j}', weight=conflict_weight)
    
    # 4e. Geographic clustering edges - connect nearby customers
    for i in range(1, N + 1):
        for j in range(i + 1, N + 1):
            distance_ij = distance_matrix[i][j] if i < len(distance_matrix) and j < len(distance_matrix[i]) else max_distance
            
            # Connect customers that are very close (might be served in same route)
            if distance_ij <= max_distance * 0.2:  # Very close customers
                proximity_weight = math.exp(-5.0 * distance_ij / max_distance)
                G.add_edge(f'customer_{i}', f'customer_{j}', weight=proximity_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()