#!/usr/bin/env python3
"""
Graph converter for vrplc_service problem.
Created using subagent_prompt.md version: v_02

This problem is about Vehicle Routing Problem with Location Congestion (VRPLC).
It extends pickup-delivery with time windows by adding location capacity constraints.
Key challenges: tight time windows, limited location capacity, pickup-delivery pairing, vehicle capacity
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the VRPLC problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model requests, vehicles, locations, and temporal/capacity constraints
    - Request nodes (pickups/deliveries) as type 0 with time urgency weights
    - Constraint nodes (capacity, time, coupling) as type 1 with tightness weights
    - Location/vehicle resource nodes as type 2 with utilization weights
    - Connect requests to constraints they participate in (bipartite structure)
    """
    # Extract problem parameters
    T = json_data.get('T', 100)      # Time horizon
    V = json_data.get('V', 1)        # Number of vehicles  
    Q = json_data.get('Q', 10)       # Vehicle capacity
    L = json_data.get('L', 1)        # Number of locations
    C = json_data.get('C', 1)        # Location capacity
    P = json_data.get('P', 1)        # Number of pickup requests
    R = 2 * P                        # Total requests (pickup + delivery)
    
    # Request data
    l = json_data.get('l', [])       # Location of each request
    a = json_data.get('a', [])       # Earliest start time
    b = json_data.get('b', [])       # Latest start time
    s = json_data.get('s', [])       # Service duration
    q = json_data.get('q', [])       # Load change (positive for pickup, negative for delivery)
    time_matrix = json_data.get('time', [])  # Travel times (flattened N x N matrix)
    
    G = nx.Graph()
    
    # 1. Request nodes (type 0) - pickups and deliveries
    max_load = max(abs(load) for load in q) if q else 1
    max_service = max(s) if s else 1
    
    for i in range(R):
        if i < len(a) and i < len(b):
            # Time window tightness (smaller window = higher urgency)
            window_size = b[i] - a[i] if b[i] > a[i] else 1
            time_urgency = 1.0 - min(window_size / T, 1.0)
            
            # Load importance (larger loads are more constraining)
            load_importance = abs(q[i]) / max_load if i < len(q) and max_load > 0 else 0.5
            
            # Service duration impact (longer service = more constraining)
            service_impact = s[i] / max_service if i < len(s) and max_service > 0 else 0.5
            
            # Combined weight using non-linear combination
            weight = math.sqrt(time_urgency * 0.5 + load_importance * 0.3 + service_impact * 0.2)
            
            G.add_node(f'request_{i}', type=0, weight=min(weight, 1.0))
    
    # 2. Vehicle resource nodes (type 2)
    total_demand = sum(abs(load) for load in q if load > 0)
    vehicle_utilization = min(total_demand / (V * Q), 1.0) if V > 0 and Q > 0 else 0.5
    
    for v in range(V):
        G.add_node(f'vehicle_{v}', type=2, weight=vehicle_utilization)
    
    # 3. Location resource nodes (type 2)
    location_requests = {}
    for i in range(min(len(l), R)):
        loc = l[i]
        if loc not in location_requests:
            location_requests[loc] = []
        location_requests[loc].append(i)
    
    for loc in range(1, L + 1):
        requests_at_loc = len(location_requests.get(loc, []))
        # Location congestion (more requests per capacity = higher weight)
        congestion = min(requests_at_loc / C, 1.0) if C > 0 else 0.5
        G.add_node(f'location_{loc}', type=2, weight=congestion)
    
    # 4. Constraint nodes (type 1)
    
    # Vehicle capacity constraints (one per vehicle)
    for v in range(V):
        # Estimate capacity tightness
        avg_load_per_vehicle = total_demand / V if V > 0 else 0
        capacity_tightness = min(avg_load_per_vehicle / Q, 1.0) if Q > 0 else 0.5
        G.add_node(f'capacity_constraint_{v}', type=1, weight=capacity_tightness)
    
    # Location capacity constraints (one per location)
    for loc in range(1, L + 1):
        requests_at_loc = location_requests.get(loc, [])
        if requests_at_loc:
            # Calculate simultaneous service potential
            overlapping_services = 0
            for i in requests_at_loc:
                for j in requests_at_loc:
                    if i != j and i < len(a) and i < len(b) and j < len(a) and j < len(b):
                        # Check if time windows could overlap
                        if not (b[i] < a[j] or b[j] < a[i]):
                            overlapping_services += 1
            
            congestion_tightness = min(overlapping_services / (2 * C), 1.0) if C > 0 else 0.5
            G.add_node(f'location_capacity_{loc}', type=1, weight=congestion_tightness)
    
    # Pickup-delivery coupling constraints
    for pickup in range(P):
        delivery = pickup + P
        if pickup < len(a) and delivery < len(a):
            # Coupling tightness based on time window alignment
            pickup_end = b[pickup] if pickup < len(b) else 0
            delivery_start = a[delivery] if delivery < len(a) else 0
            
            if pickup_end > 0 and delivery_start > 0:
                time_gap = max(delivery_start - pickup_end, 0)
                coupling_tightness = math.exp(-time_gap / T) if T > 0 else 0.5
            else:
                coupling_tightness = 0.5
                
            G.add_node(f'coupling_{pickup}', type=1, weight=coupling_tightness)
    
    # Time window constraints (individual request constraints)
    for i in range(R):
        if i < len(a) and i < len(b):
            window_size = b[i] - a[i]
            window_tightness = 1.0 - min(window_size / T, 1.0) if T > 0 else 0.5
            G.add_node(f'time_window_{i}', type=1, weight=window_tightness)
    
    # 5. Edges - Connect requests to constraints they participate in
    
    # Requests to vehicle capacity constraints (all requests affect all vehicles potentially)
    for i in range(R):
        for v in range(V):
            load_impact = abs(q[i]) / Q if i < len(q) and Q > 0 else 0.5
            G.add_edge(f'request_{i}', f'capacity_constraint_{v}', weight=min(load_impact, 1.0))
    
    # Requests to location capacity constraints
    for i in range(min(len(l), R)):
        loc = l[i]
        if loc >= 1 and loc <= L:
            service_duration = s[i] if i < len(s) else 1
            service_impact = min(service_duration / max_service, 1.0) if max_service > 0 else 0.5
            G.add_edge(f'request_{i}', f'location_capacity_{loc}', weight=service_impact)
    
    # Pickup-delivery coupling edges
    for pickup in range(P):
        delivery = pickup + P
        load_consistency = abs(q[pickup] + q[delivery]) if pickup < len(q) and delivery < len(q) else 0
        coupling_strength = 1.0 - min(load_consistency / max_load, 1.0) if max_load > 0 else 0.8
        
        G.add_edge(f'request_{pickup}', f'coupling_{pickup}', weight=coupling_strength)
        G.add_edge(f'request_{delivery}', f'coupling_{pickup}', weight=coupling_strength)
    
    # Time window constraint edges
    for i in range(R):
        G.add_edge(f'request_{i}', f'time_window_{i}', weight=1.0)
    
    # Requests to vehicle/location resources
    for i in range(R):
        # Connect to all vehicles (assignment flexibility)
        for v in range(V):
            G.add_edge(f'request_{i}', f'vehicle_{v}', weight=0.6)
        
        # Connect to location
        if i < len(l):
            loc = l[i]
            if loc >= 1 and loc <= L:
                G.add_edge(f'request_{i}', f'location_{loc}', weight=0.8)
    
    # Add conflict edges for competing requests at same location
    for loc in range(1, L + 1):
        requests_at_loc = location_requests.get(loc, [])
        if len(requests_at_loc) > C:  # Oversubscribed location
            # Add conflicts between requests with overlapping time windows
            for i in range(len(requests_at_loc)):
                for j in range(i + 1, len(requests_at_loc)):
                    req1, req2 = requests_at_loc[i], requests_at_loc[j]
                    if req1 < len(a) and req1 < len(b) and req2 < len(a) and req2 < len(b):
                        # Check time window overlap
                        if not (b[req1] < a[req2] or b[req2] < a[req1]):
                            overlap_ratio = min(
                                (min(b[req1], b[req2]) - max(a[req1], a[req2])) / 
                                max(b[req1] - a[req1], b[req2] - a[req2], 1), 1.0
                            )
                            if overlap_ratio > 0:
                                G.add_edge(f'request_{req1}', f'request_{req2}', 
                                          weight=overlap_ratio)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()