#!/usr/bin/env python3
"""
Graph converter for yumi-dynamic problem.
Created using subagent_prompt.md version: v_02

This problem is about dual-arm robotic task scheduling with collision avoidance.
YuMi robot has two arms that must coordinate to perform assembly tasks at various locations.
Key challenges: spatial collision avoidance, task ordering constraints, and multi-resource scheduling.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the yumi-dynamic problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with task-constraint structure
    - Tasks as variable nodes (type 0): different task types with varying difficulty
    - Constraints as constraint nodes (type 1): ordering, collision, resource constraints
    - Locations as resource nodes (type 2): spatial locations with arm accessibility
    - Edges model task participation in constraints and resource usage
    """
    G = nx.Graph()
    
    # Extract key data
    task_durations = json_data.get('task_durations', [])
    num_tasks = len(task_durations)
    
    output_tasks = json_data.get('OUTPUT_TASKS', [])
    fixture_task_orders = json_data.get('fixture_task_orders', [])
    gripper_pick_tasks_orders = json_data.get('gripper_pick_tasks_orders', [])
    suction_pick_tasks_orders = json_data.get('suction_pick_tasks_orders', [])
    
    left_arm_travel_times = json_data.get('left_arm_travel_times', [])
    right_arm_travel_times = json_data.get('right_arm_travel_times', [])
    
    # Estimate number of locations from travel time matrix
    if left_arm_travel_times:
        num_locations = int(math.sqrt(len(left_arm_travel_times)))
    else:
        num_locations = 10  # fallback
    
    if num_tasks == 0:
        # Empty problem, return minimal graph
        G.add_node('dummy', type=0, weight=0.5)
        return G
    
    # Calculate difficulty metrics
    max_duration = max(task_durations) if task_durations else 1
    avg_duration = sum(task_durations) / len(task_durations) if task_durations else 1
    
    # Add task nodes (type 0) with duration-based weights
    for i in range(num_tasks):
        duration = task_durations[i] if i < len(task_durations) else avg_duration
        # Non-linear weight: emphasize high-duration tasks exponentially
        difficulty = 1.0 - math.exp(-3.0 * duration / max_duration)
        G.add_node(f'task_{i+1}', type=0, weight=min(difficulty, 1.0))
    
    # Add location nodes (type 2) with accessibility-based weights
    for loc in range(num_locations):
        # Calculate accessibility: how many arms can reach this location
        left_accessible = True
        right_accessible = True
        
        if loc < num_locations and left_arm_travel_times:
            try:
                # Check diagonal element for accessibility (-1 means unreachable)
                left_index = loc * num_locations + loc
                if left_index < len(left_arm_travel_times):
                    left_accessible = left_arm_travel_times[left_index] >= 0
            except:
                pass
                
        if loc < num_locations and right_arm_travel_times:
            try:
                right_index = loc * num_locations + loc
                if right_index < len(right_arm_travel_times):
                    right_accessible = right_arm_travel_times[right_index] >= 0
            except:
                pass
        
        # Location difficulty: fewer accessible arms = higher weight
        if left_accessible and right_accessible:
            accessibility_weight = 0.3  # Both arms can reach
        elif left_accessible or right_accessible:
            accessibility_weight = 0.7  # Only one arm can reach
        else:
            accessibility_weight = 1.0  # Neither arm can reach (problematic)
            
        G.add_node(f'location_{loc}', type=2, weight=accessibility_weight)
    
    # Add constraint nodes for different constraint types
    
    # 1. Output task constraints (critical final tasks)
    for i, output_task in enumerate(output_tasks):
        if output_task > 0 and output_task <= num_tasks:
            constraint_id = f'output_constraint_{i}'
            G.add_node(constraint_id, type=1, weight=0.9)  # High importance
            G.add_edge(f'task_{output_task}', constraint_id, weight=0.9)
    
    # 2. Fixture ordering constraints (strong precedence)
    if fixture_task_orders:
        fixture_constraints = []
        current_fixture = []
        
        for task in fixture_task_orders:
            if task == -1:  # End of current fixture sequence
                if len(current_fixture) > 1:
                    fixture_constraints.append(current_fixture[:])
                current_fixture = []
            elif task > 0 and task <= num_tasks:
                current_fixture.append(task)
        
        # Add final fixture if exists
        if len(current_fixture) > 1:
            fixture_constraints.append(current_fixture)
        
        for fix_idx, fixture_seq in enumerate(fixture_constraints):
            constraint_id = f'fixture_order_{fix_idx}'
            # Weight by sequence complexity (non-linear)
            complexity = min(1.0, math.log(len(fixture_seq) + 1) / math.log(6))
            G.add_node(constraint_id, type=1, weight=complexity)
            
            # Connect all tasks in the fixture sequence
            for task in fixture_seq:
                G.add_edge(f'task_{task}', constraint_id, weight=0.8)
    
    # 3. Gripper pick-and-place constraints
    if gripper_pick_tasks_orders:
        gripper_constraints = []
        current_sequence = []
        
        for task in gripper_pick_tasks_orders:
            if task == -1:
                if len(current_sequence) >= 2:  # Pick and place pair
                    gripper_constraints.append(current_sequence[:])
                current_sequence = []
            elif task > 0 and task <= num_tasks:
                current_sequence.append(task)
        
        if len(current_sequence) >= 2:
            gripper_constraints.append(current_sequence)
        
        for grip_idx, grip_seq in enumerate(gripper_constraints):
            constraint_id = f'gripper_order_{grip_idx}'
            G.add_node(constraint_id, type=1, weight=0.7)  # Medium-high importance
            
            for task in grip_seq:
                G.add_edge(f'task_{task}', constraint_id, weight=0.75)
    
    # 4. Suction pick-and-place constraints
    if suction_pick_tasks_orders:
        suction_constraints = []
        current_sequence = []
        
        for task in suction_pick_tasks_orders:
            if task == -1:
                if len(current_sequence) >= 2:
                    suction_constraints.append(current_sequence[:])
                current_sequence = []
            elif task > 0 and task <= num_tasks:
                current_sequence.append(task)
        
        if len(current_sequence) >= 2:
            suction_constraints.append(current_sequence)
        
        for suct_idx, suct_seq in enumerate(suction_constraints):
            constraint_id = f'suction_order_{suct_idx}'
            G.add_node(constraint_id, type=1, weight=0.6)  # Medium importance
            
            for task in suct_seq:
                G.add_edge(f'task_{task}', constraint_id, weight=0.7)
    
    # 5. Dual-arm collision avoidance constraint (global)
    collision_constraint = 'collision_avoidance'
    # Weight by problem density: more tasks in same space = higher collision risk
    task_density = min(1.0, num_tasks / (num_locations * 2))  # 2 arms
    collision_weight = 0.5 + 0.5 * task_density  # Range [0.5, 1.0]
    G.add_node(collision_constraint, type=1, weight=collision_weight)
    
    # Connect all tasks to collision constraint with travel-based weights
    for i in range(num_tasks):
        # Tasks with longer durations have higher collision potential
        duration = task_durations[i] if i < len(task_durations) else avg_duration
        collision_edge_weight = min(1.0, duration / max_duration * 0.8)
        G.add_edge(f'task_{i+1}', collision_constraint, weight=collision_edge_weight)
    
    # 6. Add edges between tasks and locations (resource usage)
    # Connect tasks to locations they might use based on travel times
    for i in range(num_tasks):
        task_node = f'task_{i+1}'
        
        # Connect to accessible locations with travel-time weights
        for loc in range(num_locations):
            loc_node = f'location_{loc}'
            
            # Calculate minimum travel time to this location from either arm
            min_travel_time = float('inf')
            
            if left_arm_travel_times and loc < num_locations:
                for from_loc in range(num_locations):
                    try:
                        travel_idx = from_loc * num_locations + loc
                        if travel_idx < len(left_arm_travel_times):
                            travel_time = left_arm_travel_times[travel_idx]
                            if travel_time >= 0:  # Accessible
                                min_travel_time = min(min_travel_time, travel_time)
                    except:
                        pass
            
            if right_arm_travel_times and loc < num_locations:
                for from_loc in range(num_locations):
                    try:
                        travel_idx = from_loc * num_locations + loc
                        if travel_idx < len(right_arm_travel_times):
                            travel_time = right_arm_travel_times[travel_idx]
                            if travel_time >= 0:  # Accessible
                                min_travel_time = min(min_travel_time, travel_time)
                    except:
                        pass
            
            # Add edge if location is accessible
            if min_travel_time < float('inf'):
                # Non-linear weight: closer locations have higher weights
                max_reasonable_travel = 50  # Reasonable upper bound
                travel_weight = math.exp(-2.0 * min_travel_time / max_reasonable_travel)
                G.add_edge(task_node, loc_node, weight=min(travel_weight, 1.0))
    
    # Add some high-value task conflict edges for oversubscribed locations
    # Find tasks that compete for scarce resources
    high_duration_tasks = []
    for i in range(num_tasks):
        if i < len(task_durations):
            duration = task_durations[i]
            if duration > avg_duration * 1.5:  # High-duration tasks
                high_duration_tasks.append(i + 1)
    
    # Add conflict edges between high-duration tasks (limited pair-wise)
    for i in range(min(len(high_duration_tasks), 5)):
        for j in range(i + 1, min(len(high_duration_tasks), 5)):
            task1 = f'task_{high_duration_tasks[i]}'
            task2 = f'task_{high_duration_tasks[j]}'
            
            # Conflict weight based on combined duration
            dur1 = task_durations[high_duration_tasks[i] - 1] if high_duration_tasks[i] - 1 < len(task_durations) else avg_duration
            dur2 = task_durations[high_duration_tasks[j] - 1] if high_duration_tasks[j] - 1 < len(task_durations) else avg_duration
            conflict_weight = min(1.0, (dur1 + dur2) / (2 * max_duration))
            
            G.add_edge(task1, task2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()