# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the FLECC problem data."""
    # Initialize result dictionary with placeholder values
    results = {"README": ""}
    # Pre-fill characteristic keys
    for i in range(1, 51):
        results[f"characteristic_{i}"] = 0.0
    try:
        # Load instance data
        instance_data = input_data()
        # Extract basic parameters
        k = int(instance_data.get('nbCharacter', 0))
        L = int(instance_data.get('codeWordLength', 0))
        m = int(instance_data.get('numOfCodeWords', 0))
        min_d = float(instance_data.get('minDist', 0))
        max_d = float(instance_data.get('maxDist', 0))
        dist_list = instance_data.get('dist', [])
        # Convert to numpy array
        D = np.array(dist_list, dtype=float)
        # Compute off-diagonal distances
        mask = ~np.eye(k, dtype=bool)
        D_off = D[mask]
        n_off = D_off.size
        # Basic instance features
        total_possible = k ** L
        ratio_code_possible = m / total_possible if total_possible > 0 else 0.0
        # Statistics of distance matrix
        mean_d = float(np.mean(D_off))
        std_d = float(np.std(D_off, ddof=0))
        min_dval = float(np.min(D_off))
        max_dval = float(np.max(D_off))
        median_d = float(np.median(D_off))
        # Skewness and kurtosis
        m2 = np.mean((D_off - mean_d) ** 2)
        m3 = np.mean((D_off - mean_d) ** 3)
        m4 = np.mean((D_off - mean_d) ** 4)
        skew_d = float(m3 / (m2 ** 1.5)) if m2 > 0 else 0.0
        kurt_d = float(m4 / (m2 ** 2) - 3.0) if m2 > 0 else 0.0
        # Unique distances
        unique_vals = np.unique(D_off)
        count_eq_min = np.sum(D_off == min_d)
        count_eq_max = np.sum(D_off == max_d)
        count_ge_min = np.sum(D_off >= min_d)
        count_le_max = np.sum(D_off <= max_d)
        count_between = np.sum((D_off >= min_d) & (D_off <= max_d))
        # Graph representation of the character distance matrix
        G = nx.Graph()
        G.add_nodes_from(range(k))
        for i in range(k):
            for j in range(i+1, k):
                G.add_edge(i, j, weight=float(D[i, j]))
        # Weighted degree per node
        deg_w = np.array([d for _, d in G.degree(weight='weight')], dtype=float)
        avg_deg_w = float(np.mean(deg_w))
        std_deg_w = float(np.std(deg_w, ddof=0))
        # Minimum spanning tree metrics
        T = nx.minimum_spanning_tree(G, weight='weight')
        mst_weights = np.array([d['weight'] for u, v, d in T.edges(data=True)], dtype=float)
        mst_total = float(np.sum(mst_weights))
        mst_std = float(np.std(mst_weights, ddof=0)) if mst_weights.size > 0 else 0.0
        # Graph metrics
        density = float(nx.density(G))
        clustering = nx.clustering(G, weight='weight')
        avg_clust = float(np.mean(list(clustering.values())))
        std_clust = float(np.std(list(clustering.values()), ddof=0))
        trans = float(nx.transitivity(G))
        assort = float(nx.degree_assortativity_coefficient(G))
        # Shortest path / eccentricity / diameter / radius
        lengths = dict(nx.all_pairs_dijkstra_path_length(G, weight='weight'))
        ecc = np.array([max(lengths[n].values()) for n in G.nodes()], dtype=float)
        diam = float(np.max(ecc)) if ecc.size > 0 else 0.0
        rad = float(np.min(ecc)) if ecc.size > 0 else 0.0
        ecc_mean = float(np.mean(ecc)) if ecc.size > 0 else 0.0
        ecc_std = float(np.std(ecc, ddof=0)) if ecc.size > 0 else 0.0
        is_conn = int(nx.is_connected(G))
        n_comp = int(nx.number_connected_components(G))
        # Spectral properties of distance matrix
        eigs = np.linalg.eigvals(D)
        eigs_real = eigs.real
        eig_max = float(np.max(eigs_real))
        eig_min = float(np.min(eigs_real))
        spec_rad = float(np.max(np.abs(eigs_real)))
        eig_std = float(np.std(eigs_real, ddof=0))
        trace_D = float(np.trace(D))
        fro_norm = float(np.linalg.norm(D, ord='fro'))
        rank_D = int(np.linalg.matrix_rank(D))
        # Zero entry proportions
        total_entries = k * k
        count_zero_off = int(np.sum(D_off == 0))
        count_zero_total = int(np.sum(D == 0))
        prop_zero_off = float(count_zero_off) / n_off if n_off > 0 else 0.0
        prop_zero_total = float(count_zero_total) / total_entries if total_entries > 0 else 0.0
        # Variables and constraints counts
        total_vars = L * m
        cons_count = int(m * (m - 1) / 2)
        # Distance range
        dist_range = max_dval - min_dval
        # Fill results
        results["README"] = (
            "This instance of the Fixed Length Error Correcting Codes problem was analyzed by first converting the provided distance matrix into a weighted complete graph where nodes represent characters and edges carry their pairwise distances. "
            "Statistical metrics (mean, std, skewness, kurtosis) were computed on the off-diagonal distances to summarize the alphabet's distance distribution. "
            "Graph-theoretic measures, such as density, clustering coefficients, assortativity, and shortest-path eccentricities, were calculated using NetworkX. "
            "Spectral properties of the distance matrix, including eigenvalues and norms, capture global structural characteristics. Variables and constraint counts reflect the size of the decision space. "
            "Together, these features characterize distance diversity, graph connectivity, and instance scale, informing solver parameter tuning for performance optimization."
        )
        # Assign characteristics
        results["characteristic_1"] = k
        results["characteristic_2"] = L
        results["characteristic_3"] = m
        results["characteristic_4"] = min_d
        results["characteristic_5"] = max_d
        results["characteristic_6"] = float(total_possible)
        results["characteristic_7"] = ratio_code_possible
        results["characteristic_8"] = mean_d
        results["characteristic_9"] = std_d
        results["characteristic_10"] = min_dval
        results["characteristic_11"] = max_dval
        results["characteristic_12"] = median_d
        results["characteristic_13"] = skew_d
        results["characteristic_14"] = kurt_d
        results["characteristic_15"] = float(unique_vals.size)
        results["characteristic_16"] = float(count_eq_min) / n_off if n_off>0 else 0.0  # prop distances == minDist
        results["characteristic_17"] = float(count_eq_max) / n_off if n_off>0 else 0.0  # prop distances == maxDist
        results["characteristic_18"] = float(count_ge_min) / n_off if n_off>0 else 0.0 # prop >= minDist
        results["characteristic_19"] = float(count_le_max) / n_off if n_off>0 else 0.0 # prop <= maxDist
        results["characteristic_20"] = float(count_between) / n_off if n_off>0 else 0.0 # prop between minDist and maxDist
        results["characteristic_21"] = avg_deg_w
        results["characteristic_22"] = std_deg_w
        results["characteristic_23"] = mean_d  # edge weight mean
        results["characteristic_24"] = std_d    # edge weight std
        results["characteristic_25"] = mst_total
        results["characteristic_26"] = mst_std
        results["characteristic_27"] = density
        results["characteristic_28"] = avg_clust
        results["characteristic_29"] = std_clust
        results["characteristic_30"] = trans
        results["characteristic_31"] = assort
        results["characteristic_32"] = diam
        results["characteristic_33"] = ecc_mean
        results["characteristic_34"] = ecc_std
        results["characteristic_35"] = rad
        results["characteristic_36"] = is_conn
        results["characteristic_37"] = n_comp
        results["characteristic_38"] = eig_max
        results["characteristic_39"] = eig_min
        results["characteristic_40"] = spec_rad
        results["characteristic_41"] = eig_std
        results["characteristic_42"] = trace_D
        results["characteristic_43"] = fro_norm
        results["characteristic_44"] = float(rank_D)
        results["characteristic_45"] = prop_zero_off
        results["characteristic_46"] = prop_zero_total
        results["characteristic_47"] = float(total_vars)
        results["characteristic_48"] = float(cons_count)
        results["characteristic_49"] = dist_range
        results["characteristic_50"] = float(count_zero_total) / total_entries if total_entries>0 else 0.0  # diag zeros proportion
    except Exception as e:
        results["error"] = str(e)
    # Output results
    output_results(results)

if __name__ == "__main__":
    main()
