# Import helper functions for input/output
# IMPORTANT: You MUST import these functions at the beginning of your script
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the problem data."""
    # Get the instance data using the helper function
    instance_data = input_data()

    # Extract basic parameters from the instance
    n_cars = instance_data.get('n_cars', 0)
    n_classes = instance_data.get('n_classes', 0)
    n_options = instance_data.get('n_options', 0)
    quantity = np.array(instance_data.get('quantity', []), dtype=float)
    maxcars = np.array(instance_data.get('maxcars', []), dtype=float)
    blksize_delta = np.array(instance_data.get('blksize_delta', []), dtype=float)
    usage = np.array(instance_data.get('usage', []), dtype=int)

    # Compute derived metrics for window sizes
    window_sizes = maxcars + blksize_delta
    tightness = np.divide(maxcars, window_sizes, out=np.zeros_like(maxcars), where=window_sizes!=0)

    # Build bipartite graph: classes U options V
    B = nx.Graph()
    class_nodes = [f"c_{i}" for i in range(n_classes)]
    option_nodes = [f"o_{j}" for j in range(n_options)]
    B.add_nodes_from(class_nodes, bipartite=0)
    B.add_nodes_from(option_nodes, bipartite=1)
    edges = []
    for i in range(n_classes):
        for j in range(n_options):
            if usage[i, j] == 1:
                edges.append((class_nodes[i], option_nodes[j]))
    B.add_edges_from(edges)

    # Project to class graph (classes connected if share an option)
    class_graph = nx.bipartite.projected_graph(B, class_nodes)

    # Helper to compute skewness and kurtosis
    def skew(arr):
        m = np.mean(arr)
        s = np.std(arr)
        return np.mean((arr - m)**3) / s**3 if s > 0 else 0.0
    def kurt(arr):
        m = np.mean(arr)
        s = np.std(arr)
        return np.mean((arr - m)**4) / s**4 - 3.0 if s > 0 else 0.0

    # Prepare results dictionary
    results = {
        "README": (
            "This instance was analyzed by parsing the Car Sequencing JSON schema, constructing a bipartite graph "
            "with class and option nodes, and computing both statistical and structural graph metrics. We extracted "
            "problem size parameters (number of cars, classes, options), distributional statistics for class quantities, "
            "option capacities, and sliding window sizes. Tightness ratios quantify constraint density. Usage incidence "
            "is summarized by sparsity and per-row/-column statistics. The bipartite graph analysis provides edge counts "
            "and density, while a projected class graph reveals connectivity patterns, clustering coefficients, shortest-path "
            "characteristics, component structure, and spectral properties. These 50 standardized characteristics capture "
            "the complexity, scale, and tightness of the instance, informing solver parameter tuning by highlighting key "
            "structural features such as constraint tightness, variable interdependencies, and graph connectivity patterns."
        ),
        "characteristic_1": float(n_cars),               # total cars
        "characteristic_2": float(n_classes),            # number of classes
        "characteristic_3": float(n_options),            # number of options
        "characteristic_4": float(np.sum(quantity)),    # sum of quantities
        "characteristic_5": float(np.mean(quantity)),   # mean quantity per class
        "characteristic_6": float(np.std(quantity)),    # std quantity per class
        "characteristic_7": float(np.min(quantity)),    # min quantity
        "characteristic_8": float(np.max(quantity)),    # max quantity
        "characteristic_9": float(skew(quantity)),      # skewness of quantity distribution
        "characteristic_10": float(kurt(quantity)),     # kurtosis of quantity distribution
        "characteristic_11": float(np.sum(maxcars)),   # sum of maxcars
        "characteristic_12": float(np.mean(maxcars)),  # mean maxcars
        "characteristic_13": float(np.std(maxcars)),   # std maxcars
        "characteristic_14": float(np.min(maxcars)),   # min maxcars
        "characteristic_15": float(np.max(maxcars)),   # max maxcars
        "characteristic_16": float(np.sum(blksize_delta)),  # sum of block size deltas
        "characteristic_17": float(np.mean(blksize_delta)),# mean delta
        "characteristic_18": float(np.std(blksize_delta)), # std delta
        "characteristic_19": float(np.min(blksize_delta)), # min delta
        "characteristic_20": float(np.max(blksize_delta)), # max delta
        "characteristic_21": float(np.mean(window_sizes)),  # mean window size
        "characteristic_22": float(np.std(window_sizes)),   # std window size
        "characteristic_23": float(np.min(window_sizes)),   # min window size
        "characteristic_24": float(np.max(window_sizes)),   # max window size
        "characteristic_25": float(np.mean(tightness)),     # mean tightness ratio
        "characteristic_26": float(np.std(tightness)),      # std tightness
        "characteristic_27": float(np.max(tightness)),      # max tightness
        "characteristic_28": float(np.min(tightness)),      # min tightness
        "characteristic_29": float(np.sum(usage) / (n_classes * n_options) if n_classes*n_options>0 else 0.0), # usage density
        "characteristic_30": float(np.mean(np.sum(usage, axis=1))), # mean ones per class
        "characteristic_31": float(np.std(np.sum(usage, axis=1))),  # std ones per class
        "characteristic_32": float(np.min(np.sum(usage, axis=1))),  # min ones per class
        "characteristic_33": float(np.max(np.sum(usage, axis=1))),  # max ones per class
        "characteristic_34": float(np.mean(np.sum(usage, axis=0))), # mean ones per option
        "characteristic_35": float(np.std(np.sum(usage, axis=0))),  # std ones per option
        "characteristic_36": float(np.min(np.sum(usage, axis=0))),  # min ones per option
        "characteristic_37": float(np.max(np.sum(usage, axis=0))),  # max ones per option
        "characteristic_38": float(B.number_of_nodes()),            # bipartite graph nodes
        "characteristic_39": float(B.number_of_edges()),            # bipartite graph edges
        "characteristic_40": float(nx.density(B)),                  # bipartite density
        "characteristic_41": float(class_graph.number_of_edges()),  # class graph edges
        "characteristic_42": float(nx.density(class_graph)),        # class graph density
        "characteristic_43": float(np.mean([d for _,d in class_graph.degree()])), # avg degree
        "characteristic_44": float(nx.average_clustering(class_graph)),           # clustering coeff
        "characteristic_45": float(nx.average_shortest_path_length(class_graph)) if nx.is_connected(class_graph) else 0.0, # avg shortest path
        "characteristic_46": float(nx.diameter(class_graph)) if nx.is_connected(class_graph) else 0.0,                  # diameter
        "characteristic_47": float(nx.number_connected_components(class_graph)),                                            # n components
        "characteristic_48": float(max(len(c) for c in nx.connected_components(class_graph))),                             # size largest comp
        "characteristic_49": float(nx.degree_assortativity_coefficient(class_graph)),                                 # assortativity
        "characteristic_50": float(max(abs(np.linalg.eigvals(nx.to_numpy_array(class_graph)))))                            # spectral radius
    }

    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
