# Import helper functions for input/output
# IMPORTANT: You MUST import these functions at the beginning of your script
# DO NOT define your own versions of these functions!
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np

def main():
    """Extract instance characteristics from the problem data."""
    # Get the instance data using the helper function
    instance_data = input_data()

    # Parse instance data
    n_cars = instance_data.get('n_cars', 0)
    n_classes = instance_data.get('n_classes', 0)
    n_options = instance_data.get('n_options', 0)
    quantity = np.array(instance_data.get('quantity', []), dtype=float)
    maxcars = np.array(instance_data.get('maxcars', []), dtype=float)
    blksize_delta = np.array(instance_data.get('blksize_delta', []), dtype=float)
    usage = np.array(instance_data.get('usage', []), dtype=int)

    # Precompute statistics
    # Quantity stats
    total_quantity = quantity.sum() if quantity.size > 0 else 0.0
    avg_quantity = quantity.mean() if quantity.size > 0 else 0.0
    std_quantity = quantity.std(ddof=0) if quantity.size > 0 else 0.0
    min_quantity = quantity.min() if quantity.size > 0 else 0.0
    max_quantity = quantity.max() if quantity.size > 0 else 0.0
    # Skewness and kurtosis
    if std_quantity > 0:
        skew_quantity = ((quantity - avg_quantity)**3).mean() / (std_quantity**3)
        kurt_quantity = ((quantity - avg_quantity)**4).mean() / (std_quantity**4) - 3.0
    else:
        skew_quantity = 0.0
        kurt_quantity = 0.0

    # Maxcars and blksize_delta stats
    avg_maxcars = maxcars.mean() if maxcars.size > 0 else 0.0
    std_maxcars = maxcars.std(ddof=0) if maxcars.size > 0 else 0.0
    min_maxcars = maxcars.min() if maxcars.size > 0 else 0.0
    max_maxcars = maxcars.max() if maxcars.size > 0 else 0.0
    avg_delta = blksize_delta.mean() if blksize_delta.size > 0 else 0.0
    std_delta = blksize_delta.std(ddof=0) if blksize_delta.size > 0 else 0.0
    min_delta = blksize_delta.min() if blksize_delta.size > 0 else 0.0
    max_delta = blksize_delta.max() if blksize_delta.size > 0 else 0.0

    # Sliding window sizes and constraints
    window_sizes = maxcars + blksize_delta
    avg_window = window_sizes.mean() if window_sizes.size > 0 else 0.0
    std_window = window_sizes.std(ddof=0) if window_sizes.size > 0 else 0.0
    min_window = window_sizes.min() if window_sizes.size > 0 else 0.0
    max_window = window_sizes.max() if window_sizes.size > 0 else 0.0
    # Constraints per option: windows of size w sliding over n_cars slots
    constraints_per_option = np.array([max(0, n_cars + 1 - w) for w in window_sizes], dtype=float)
    total_window_constraints = constraints_per_option.sum()
    avg_constraints_opt = constraints_per_option.mean() if constraints_per_option.size > 0 else 0.0
    std_constraints_opt = constraints_per_option.std(ddof=0) if constraints_per_option.size > 0 else 0.0
    min_constraints_opt = constraints_per_option.min() if constraints_per_option.size > 0 else 0.0
    max_constraints_opt = constraints_per_option.max() if constraints_per_option.size > 0 else 0.0

    # Usage stats
    total_usages = usage.sum() if usage.size > 0 else 0
    usage_per_class = usage.sum(axis=1) if usage.size > 0 else np.array([])
    usage_per_option = usage.sum(axis=0) if usage.size > 0 else np.array([])
    avg_upc = usage_per_class.mean() if usage_per_class.size > 0 else 0.0
    std_upc = usage_per_class.std(ddof=0) if usage_per_class.size > 0 else 0.0
    min_upc = usage_per_class.min() if usage_per_class.size > 0 else 0.0
    max_upc = usage_per_class.max() if usage_per_class.size > 0 else 0.0
    avgupo = usage_per_option.mean() if usage_per_option.size > 0 else 0.0
    stdupo = usage_per_option.std(ddof=0) if usage_per_option.size > 0 else 0.0
    minupo = usage_per_option.min() if usage_per_option.size > 0 else 0.0
    maxupo = usage_per_option.max() if usage_per_option.size > 0 else 0.0

    # Build class graph: classes connected if share an option
    Gc = nx.Graph()
    Gc.add_nodes_from(range(n_classes))
    for i in range(n_classes):
        for j in range(i+1, n_classes):
            if np.any(usage[i] & usage[j]):
                Gc.add_edge(i, j)
    # Graph metrics for class graph
    cg_density = nx.density(Gc)
    degrees = np.array([d for _, d in Gc.degree()])
    cg_avg_degree = degrees.mean() if degrees.size>0 else 0.0
    cg_clustering = nx.average_clustering(Gc) if n_classes>0 else 0.0
    cg_components = nx.number_connected_components(Gc)
    # Largest component metrics
    if n_classes > 0 and Gc.number_of_nodes()>0 and nx.is_connected(Gc):
        comp_sub = Gc
    elif n_classes>0 and Gc.number_of_nodes()>0:
        largest = max(nx.connected_components(Gc), key=len)
        comp_sub = Gc.subgraph(largest)
    else:
        comp_sub = Gc
    try:
        cg_diameter = nx.diameter(comp_sub) if comp_sub.number_of_nodes()>0 else 0
        cg_avg_sp = nx.average_shortest_path_length(comp_sub) if comp_sub.number_of_nodes()>1 else 0.0
    except Exception:
        cg_diameter = 0
        cg_avg_sp = 0.0

    # Build option graph: options connected if they co-occur in any class
    Go = nx.Graph()
    Go.add_nodes_from(range(n_options))
    for i in range(n_options):
        for j in range(i+1, n_options):
            if np.any(usage[:, i] & usage[:, j]):
                Go.add_edge(i, j)
    og_density = nx.density(Go)
    o_degrees = np.array([d for _, d in Go.degree()])
    og_avg_degree = o_degrees.mean() if o_degrees.size>0 else 0.0
    og_clustering = nx.average_clustering(Go) if n_options>0 else 0.0
    og_components = nx.number_connected_components(Go)
    if n_options>0 and Go.number_of_nodes()>0 and nx.is_connected(Go):
        o_sub = Go
    elif n_options>0 and Go.number_of_nodes()>0:
        largest_o = max(nx.connected_components(Go), key=len)
        o_sub = Go.subgraph(largest_o)
    else:
        o_sub = Go
    try:
        og_diameter = nx.diameter(o_sub) if o_sub.number_of_nodes()>0 else 0
        og_avg_sp = nx.average_shortest_path_length(o_sub) if o_sub.number_of_nodes()>1 else 0.0
    except Exception:
        og_diameter = 0
        og_avg_sp = 0.0

    # Prepare results dictionary
    results = {
        "README": (
            "This instance was analyzed by parsing the car sequencing problem data from JSON, "
            "and computing descriptive statistics and graph-based metrics to capture structural complexity. "
            "Numeric arrays for quantities, option capacities, block size deltas, and usage incidence "
            "were processed with NumPy to extract means, variances, and higher moments (skewness and kurtosis). "
            "Sliding-window constraints were quantified by computing window sizes per option and the total number "
            "of such constraints. The usage matrix was analyzed to measure option distribution across classes, "
            "and a bipartite incidence structure between classes and options was projected into two graphs: one "
            "connecting classes that share options, and another connecting options co-occurring in any class. "
            "NetworkX was used to calculate graph density, degree statistics, clustering coefficients, component counts, "
            "diameters, and average shortest path lengths, capturing interaction topology. These features are relevant "
            "as they reflect problem size, constraint tightness, sparsity, and structural regularity, which influence "
            "constraint solver performance for car sequencing models."
        ),
        "characteristic_1": float(n_cars),                     # total number of cars
        "characteristic_2": float(n_classes),                  # number of classes
        "characteristic_3": float(n_options),                  # number of options
        "characteristic_4": float(total_quantity),             # sum of quantities
        "characteristic_5": float(avg_quantity),               # avg cars per class
        "characteristic_6": float(std_quantity),               # std of cars per class
        "characteristic_7": float(min_quantity),               # min cars in a class
        "characteristic_8": float(max_quantity),               # max cars in a class
        "characteristic_9": float(skew_quantity),              # skewness of class quantities
        "characteristic_10": float(kurt_quantity),             # kurtosis of class quantities
        "characteristic_11": float(avg_maxcars),               # avg maxcars per option
        "characteristic_12": float(std_maxcars),               # std maxcars per option
        "characteristic_13": float(min_maxcars),               # min maxcars
        "characteristic_14": float(max_maxcars),               # max maxcars
        "characteristic_15": float(avg_delta),                 # avg block size delta
        "characteristic_16": float(std_delta),                 # std block size delta
        "characteristic_17": float(min_delta),                 # min block size delta
        "characteristic_18": float(max_delta),                 # max block size delta
        "characteristic_19": float(avg_window),                # avg sliding window size
        "characteristic_20": float(std_window),                # std sliding window size
        "characteristic_21": float(min_window),                # min sliding window size
        "characteristic_22": float(max_window),                # max sliding window size
        "characteristic_23": float(total_window_constraints),  # total window constraints
        "characteristic_24": float(avg_constraints_opt),       # avg constraints per option
        "characteristic_25": float(std_constraints_opt),       # std constraints per option
        "characteristic_26": float(min_constraints_opt),       # min constraints per option
        "characteristic_27": float(max_constraints_opt),       # max constraints per option
        "characteristic_28": float(total_usages),              # total usage incidences
        "characteristic_29": float(avg_upc),                   # avg usage per class
        "characteristic_30": float(std_upc),                   # std usage per class
        "characteristic_31": float(min_upc),                   # min usage per class
        "characteristic_32": float(max_upc),                   # max usage per class
        "characteristic_33": float(avgupo),                    # avg usage per option
        "characteristic_34": float(stdupo),                    # std usage per option
        "characteristic_35": float(minupo),                    # min usage per option
        "characteristic_36": float(maxupo),                    # max usage per option
        "characteristic_37": float(cg_density),                # class graph density
        "characteristic_38": float(cg_avg_degree),            # class graph avg degree
        "characteristic_39": float(cg_clustering),            # class graph clustering coeff
        "characteristic_40": float(cg_components),            # class graph components
        "characteristic_41": float(cg_diameter),              # class graph diameter
        "characteristic_42": float(cg_avg_sp),                # class graph avg shortest path
        "characteristic_43": float(og_density),                # option graph density
        "characteristic_44": float(og_avg_degree),            # option graph avg degree
        "characteristic_45": float(og_clustering),            # option graph clustering coeff
        "characteristic_46": float(og_components),            # option graph components
        "characteristic_47": float(og_diameter),              # option graph diameter
        "characteristic_48": float(og_avg_sp),                # option graph avg shortest path
        "characteristic_49": float(total_usages / (n_classes * n_options)) if n_classes>0 and n_options>0 else 0.0,  # usage density
        "characteristic_50": float(n_classes + total_window_constraints)  # total constraints
    }
    # Return the results using the helper function
    output_results(results)

if __name__ == "__main__":
    main()