# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the Car Sequencing problem instance."""
    # Get the instance data using the helper function
    instance_data = input_data()

    # Initialize results dictionary with required structure
    results = {'README': '',
               }
    # Pre-fill the 50 characteristic slots with zeros
    for i in range(1, 51):
        results[f'characteristic_{i}'] = 0.0

    # Prepare README description (approx. 200 words)
    readme_text = (
        "This Car Sequencing instance was analyzed by parsing the provided JSON schema, "
        "which specifies the number of cars, classes, options, quantities, sliding window constraints, and usage matrix. "
        "First, fundamental parameters such as n_cars, n_classes, and n_options were extracted. "
        "Quantitative distributions for class quantities and option constraints were statistically summarized using NumPy, "
        "including measures of central tendency, dispersion, and range. "
        "Next, a bipartite graph was constructed in NetworkX where nodes represent car classes and options, "
        "and edges indicate requirement usage. Graph-theoretic metrics such as degree distributions, density, clustering coefficient, "
        "connected components, and centrality measures were computed to capture structural complexity. "
        "Additionally, a sliding-window analysis assessed the tightness of each option's capacity constraint by examining the ratio of maxcars to window size. "
        "Finally, problem-specific features such as average options per car, option saturation, and variance in constraint tightness were calculated. "
        "These characteristics are essential for guiding solver parameter settings based on instance size, constraint tightness, and structural regularity."
    )
    results['README'] = readme_text

    try:
        # Extract basic parameters
        n_cars = instance_data.get('n_cars', 0)
        n_classes = instance_data.get('n_classes', 0)
        n_options = instance_data.get('n_options', 0)
        quantity = np.array(instance_data.get('quantity', []), dtype=float)
        maxcars = np.array(instance_data.get('maxcars', []), dtype=float)
        blksize_delta = np.array(instance_data.get('blksize_delta', []), dtype=float)
        usage = np.array(instance_data.get('usage', []), dtype=int)

        # Characteristic 1: total number of cars
        results['characteristic_1'] = n_cars
        # Characteristic 2: number of classes
        results['characteristic_2'] = n_classes
        # Characteristic 3: number of options
        results['characteristic_3'] = n_options

        # Stats on class quantities
        results['characteristic_4'] = float(np.sum(quantity))              # total quantity sum
        results['characteristic_5'] = float(np.mean(quantity))             # avg cars per class
        results['characteristic_6'] = float(np.std(quantity))              # std of class quantities
        results['characteristic_7'] = float(np.min(quantity))              # min class quantity
        results['characteristic_8'] = float(np.max(quantity))              # max class quantity
        results['characteristic_9'] = float( np.median(quantity) )         # median class quantity
        results['characteristic_10'] = float( np.percentile(quantity, 25) )# 25th percentile
        results['characteristic_11'] = float( np.percentile(quantity, 75) )# 75th percentile
        results['characteristic_12'] = float( np.ptp(quantity) )           # range of quantities

        # Stats on window sizes
        window_sizes = maxcars + blksize_delta
        results['characteristic_13'] = float(np.mean(window_sizes))        # avg window size
        results['characteristic_14'] = float(np.std(window_sizes))         # std window size
        results['characteristic_15'] = float(np.min(window_sizes))         # min window size
        results['characteristic_16'] = float(np.max(window_sizes))         # max window size
        results['characteristic_17'] = float(np.mean(maxcars / window_sizes))  # avg tightness
        results['characteristic_18'] = float(np.std(maxcars / window_sizes))   # std tightness

        # Usage matrix stats
        total_edges = np.sum(usage)
        results['characteristic_19'] = float(total_edges)                  # total usage edges
        bipartite_graph = nx.Graph()
        # Add class nodes prefixed 'C' and option nodes prefixed 'O'
        for i in range(n_classes): bipartite_graph.add_node(f'C{i}', bipartite=0)
        for j in range(n_options): bipartite_graph.add_node(f'O{j}', bipartite=1)
        # Add edges for usage
        for i in range(n_classes):
            for j in range(n_options):
                if usage[i, j] == 1:
                    bipartite_graph.add_edge(f'C{i}', f'O{j}')
        # Characteristic 20: bipartite edges
        results['characteristic_20'] = bipartite_graph.number_of_edges()
        # Characteristic 21: bipartite density
        results['characteristic_21'] = nx.density(bipartite_graph)
        # Characteristic 22: number of connected components
        results['characteristic_22'] = nx.number_connected_components(bipartite_graph)
        # Characteristic 23: largest component size
        components = list(nx.connected_components(bipartite_graph))
        largest_cc = max(len(c) for c in components) if components else 0
        results['characteristic_23'] = largest_cc
        # Characteristic 24: avg degree
        degrees = [d for n, d in bipartite_graph.degree()]
        results['characteristic_24'] = float(np.mean(degrees))
        # Characteristic 25: std degree
        results['characteristic_25'] = float(np.std(degrees))
        # Characteristic 26: avg clustering (should be zero for bipartite)
        results['characteristic_26'] = nx.average_clustering(bipartite_graph)
        # Characteristic 27: degree centrality mean
        dc = nx.degree_centrality(bipartite_graph)
        results['characteristic_27'] = float(np.mean(list(dc.values())))
        # Characteristic 28: degree centrality std
        results['characteristic_28'] = float(np.std(list(dc.values())))

        # Problem-specific: avg options per class
        options_per_class = np.sum(usage, axis=1)
        results['characteristic_29'] = float(np.mean(options_per_class))
        results['characteristic_30'] = float(np.std(options_per_class))
        results['characteristic_31'] = float(np.min(options_per_class))
        results['characteristic_32'] = float(np.max(options_per_class))

        # Problem-specific: avg classes per option
        classes_per_option = np.sum(usage, axis=0)
        results['characteristic_33'] = float(np.mean(classes_per_option))
        results['characteristic_34'] = float(np.std(classes_per_option))
        results['characteristic_35'] = float(np.min(classes_per_option))
        results['characteristic_36'] = float(np.max(classes_per_option))

        # Constraint tightness distribution percentiles
        tightness = maxcars / window_sizes
        results['characteristic_37'] = float(np.percentile(tightness, 25))
        results['characteristic_38'] = float(np.percentile(tightness, 50))
        results['characteristic_39'] = float(np.percentile(tightness, 75))
        results['characteristic_40'] = float(np.ptp(tightness))

        # Constraint slackness: window_sizes - maxcars
        slack = window_sizes - maxcars
        results['characteristic_41'] = float(np.mean(slack))
        results['characteristic_42'] = float(np.std(slack))
        results['characteristic_43'] = float(np.min(slack))
        results['characteristic_44'] = float(np.max(slack))

        # Overall usage proportion (# usage / (n_classes*n_options))
        results['characteristic_45'] = float(total_edges / (n_classes * n_options))
        # Sparsity = 1 - usage proportion
        results['characteristic_46'] = 1.0 - results['characteristic_45']

        # Connected component sizes statistics
        comp_sizes = [len(c) for c in components]
        results['characteristic_47'] = float(np.mean(comp_sizes))
        results['characteristic_48'] = float(np.std(comp_sizes))
        results['characteristic_49'] = float(np.min(comp_sizes))
        results['characteristic_50'] = float(np.max(comp_sizes))

    except Exception as e:
        # In case of error, record it
        results['error'] = str(e)

    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
