# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np

"""Car Sequencing instance characteristic extractor
This script reads a JSON-formatted car sequencing instance via `input_data()` and
computes 50 numeric characteristics describing its size, option-class structure
and theoretical difficulty.  A bipartite graph is built where nodes are car
classes and options with edges representing usage requirements.  Classic graph
metrics (density, centrality, clustering) are extracted together with window
parameters, usage sparsity and statistical moments.  The results are returned in
a dictionary whose first key is README (~200 words) followed by
characteristic_1 … characteristic_50.
"""

# ------------------------------ helpers ------------------------------

def build_bipartite_graph(quantity, usage):
    """Return NetworkX Graph with two partitions (classes, options)."""
    n_classes = len(quantity)
    n_options = len(usage[0]) if n_classes else 0
    G = nx.Graph()
    # Add class nodes prefixed with 'c' and option nodes with 'o'
    for c in range(n_classes):
        G.add_node(f"c{c}", bipartite=0, qty=quantity[c])
    for o in range(n_options):
        G.add_node(f"o{o}", bipartite=1)
    # Add edges where usage=1
    for c in range(n_classes):
        for o in range(n_options):
            if usage[c][o]:
                G.add_edge(f"c{c}", f"o{o}")
    return G


def compute_graph_metrics(G):
    """Compute several graph metrics on bipartite graph G."""
    num_nodes = G.number_of_nodes()
    num_edges = G.number_of_edges()
    possible_edges = (
        len([n for n, d in G.nodes(data=True) if d["bipartite"] == 0])
        * len([n for n, d in G.nodes(data=True) if d["bipartite"] == 1])
    )
    density = num_edges / possible_edges if possible_edges else 0.0
    degrees = np.array([deg for _, deg in G.degree()])
    avg_deg = degrees.mean() if degrees.size else 0.0
    std_deg = degrees.std(ddof=0) if degrees.size else 0.0
    max_deg = degrees.max() if degrees.size else 0.0
    min_deg = degrees.min() if degrees.size else 0.0
    # Clustering coefficient in bipartite graphs is zero, but we can project
    clustering = 0.0
    # Compute eigenvector centrality (might fail for disconnected graph)
    try:
        eigen_centrality = nx.eigenvector_centrality_numpy(G)
        eigen_vals = np.array(list(eigen_centrality.values()))
        avg_eigen = float(eigen_vals.mean())
        std_eigen = float(eigen_vals.std(ddof=0))
        max_eigen = float(eigen_vals.max())
    except Exception:
        avg_eigen = std_eigen = max_eigen = 0.0
    return {
        "num_nodes": num_nodes,
        "num_edges": num_edges,
        "density": density,
        "avg_deg": avg_deg,
        "std_deg": std_deg,
        "max_deg": max_deg,
        "min_deg": min_deg,
        "avg_eigen": avg_eigen,
        "std_eigen": std_eigen,
        "max_eigen": max_eigen,
    }


def main():
    inst = input_data()

    # Extract basic arrays
    n_cars = inst.get("n_cars", 0)
    n_classes = inst.get("n_classes", 0)
    n_options = inst.get("n_options", 0)
    quantity = np.array(inst.get("quantity", []), dtype=int)
    maxcars = np.array(inst.get("maxcars", []), dtype=int)
    blksize_delta = np.array(inst.get("blksize_delta", []), dtype=int)
    usage = np.array(inst.get("usage", []), dtype=int)

    # Derived metrics
    total_quantity = quantity.sum()
    assert (
        n_cars == total_quantity
    ), "Quantity sum does not equal n_cars – malformed instance"
    option_window = maxcars + blksize_delta  # window sizes

    # Sparsity measures
    usage_nonzero = usage.sum()
    usage_total = n_classes * n_options
    usage_density = usage_nonzero / usage_total if usage_total else 0.0

    # Per-option demand (how many cars require each option)
    option_demand = usage.T.dot(quantity)  # shape (n_options,)

    # Sliding-window tightness ratio: max allowed / window size
    tightness_ratio = maxcars / option_window

    # Build bipartite graph and compute metrics
    G = build_bipartite_graph(quantity, usage)
    g_metrics = compute_graph_metrics(G)

    # Create README (~200 words)
    readme_text = (
        "This extractor builds a bipartite graph with one partition for car "
        "classes (weighted by required quantity) and another for options; an "
        "edge indicates that the class uses the option.  From this graph we "
        "compute size, density, degree distribution and eigenvector "
        "centrality statistics which reflect how intertwined option "
        "constraints are.  Additionally the script analyses window "
        "constraints that govern each option: the raw limits, window sizes, "
        "and their tightness ratios, together with demand statistics such as "
        "average demand per option and coefficient of variation.  Usage matrix "
        "sparsity and per-class option counts capture structural regularity, "
        "while imbalance indicators assess production symmetry.  These 50 "
        "numeric characteristics collectively summarise problem size "
        "(cars, classes, options), constraint graph complexity and expected "
        "search difficulty, providing a comprehensive signature for automatic "
        "solver configuration.  All metrics are inexpensive to compute: only "
        "basic numpy operations and NetworkX analysis are used, ensuring the "
        "extractor remains lightweight.  The chosen parameters follow common "
        "portfolio-solver feature conventions and can feed machine-learning "
        "models that predict suitable branching heuristics, restart policies "
        "or propagation strengths for unseen Car Sequencing instances."
    )

    # Assemble results with 50 characteristics
    results = {
        "README": readme_text
    }

    # Characteristic values (fill sequentially)
    char_vals = [
        n_cars,  # 1 total cars
        n_classes,  # 2 total classes
        n_options,  # 3 total options
        float(quantity.mean()) if n_classes else 0.0,  # 4 avg cars per class
        float(quantity.std(ddof=0)) if n_classes else 0.0,  # 5 std cars per class
        int(quantity.max()) if quantity.size else 0,  # 6 max cars per class
        int(quantity.min()) if quantity.size else 0,  # 7 min cars per class
        usage_nonzero,  # 8 number of 1s in usage matrix
        usage_density,  # 9 usage density
        float(usage.sum(axis=1).mean()) if n_classes else 0.0,  # 10 avg options per class
        float(usage.sum(axis=1).std(ddof=0)) if n_classes else 0.0,  # 11 std options per class
        int(usage.sum(axis=1).max()) if n_classes else 0,  # 12 max options required by a class
        int(usage.sum(axis=1).min()) if n_classes else 0,  # 13 min options required by a class
        float(usage.sum(axis=0).mean()) if n_options else 0.0,  # 14 avg classes per option
        float(usage.sum(axis=0).std(ddof=0)) if n_options else 0.0,  # 15 std classes per option
        int(usage.sum(axis=0).max()) if n_options else 0,  # 16 max classes needing an option
        int(usage.sum(axis=0).min()) if n_options else 0,  # 17 min classes needing an option
        g_metrics["num_nodes"],  # 18 graph nodes
        g_metrics["num_edges"],  # 19 graph edges
        g_metrics["density"],  # 20 bipartite density
        g_metrics["avg_deg"],  # 21 avg degree
        g_metrics["std_deg"],  # 22 deg std
        g_metrics["max_deg"],  # 23 max degree
        g_metrics["min_deg"],  # 24 min degree
        g_metrics["avg_eigen"],  # 25 avg eigen centrality
        g_metrics["std_eigen"],  # 26 std eigen centrality
        g_metrics["max_eigen"],  # 27 max eigen centrality
        float(maxcars.mean()) if n_options else 0.0,  # 28 avg maxcars limit
        float(maxcars.std(ddof=0)) if n_options else 0.0,  # 29 std maxcars
        int(maxcars.max()) if maxcars.size else 0,  # 30 max of maxcars
        int(maxcars.min()) if maxcars.size else 0,  # 31 min of maxcars
        float(option_window.mean()) if n_options else 0.0,  # 32 avg window size
        float(option_window.std(ddof=0)) if n_options else 0.0,  # 33 std window size
        int(option_window.max()) if option_window.size else 0,  # 34 max window
        int(option_window.min()) if option_window.size else 0,  # 35 min window
        float(tightness_ratio.mean()) if n_options else 0.0,  # 36 avg tightness ratio
        float(tightness_ratio.std(ddof=0)) if n_options else 0.0,  # 37 std tightness ratio
        float(tightness_ratio.max()) if tightness_ratio.size else 0.0,  # 38 max tightness
        float(tightness_ratio.min()) if tightness_ratio.size else 0.0,  # 39 min tightness
        int(option_demand.sum()),  # 40 total option demand (>= cars * avg options per car)
        float(option_demand.mean()) if n_options else 0.0,  # 41 avg demand per option
        float(option_demand.std(ddof=0)) if n_options else 0.0,  # 42 std demand per option
        int(option_demand.max()) if option_demand.size else 0,  # 43 max demand option
        int(option_demand.min()) if option_demand.size else 0,  # 44 min demand option
        float((quantity.max() - quantity.min()) / quantity.mean()) if n_classes else 0.0,  # 45 class qty imbalance
        float((usage.sum(axis=0).max() - usage.sum(axis=0).min()) / n_classes) if n_options else 0.0,  # 46 option popularity imbalance
        float(np.count_nonzero(quantity == quantity.max()) / n_classes) if n_classes else 0.0,  # 47 pct dominant classes
        float(np.count_nonzero(usage.sum(axis=1) == usage.sum(axis=1).max()) / n_classes) if n_classes else 0.0,  # 48 pct classes with max options
        float(np.count_nonzero(usage.sum(axis=0) == usage.sum(axis=0).max()) / n_options) if n_options else 0.0,  # 49 pct most popular options
        g_metrics["density"] * float(tightness_ratio.mean())  # 50 combined density-tightness indicator
    ]

    # Ensure we have exactly 50 characteristics
    assert len(char_vals) == 50, "Expected 50 characteristic values"
    for idx, val in enumerate(char_vals, start=1):
        results[f"characteristic_{idx}"] = float(val)

    output_results(results)


if __name__ == "__main__":
    main()
