# Import helper functions for input/output
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np
import math


def shannon_entropy(probabilities):
    """Compute Shannon entropy (base 2)."""
    probs = np.array(probabilities, dtype=float)
    probs = probs[probs > 0]
    if probs.size == 0:
        return 0.0
    return -np.sum(probs * np.log2(probs))


def gini_coefficient(values):
    """Compute Gini coefficient for a 1-D array-like of non-negative numbers."""
    x = np.array(values, dtype=float)
    if x.size == 0:
        return 0.0
    if np.all(x == 0):
        return 0.0
    x_sorted = np.sort(x)
    n = x_sorted.size
    cumulative = np.cumsum(x_sorted)
    gini = (n + 1 - 2 * np.sum(cumulative) / cumulative[-1]) / n
    return gini


def safe_div(a, b):
    """Return a / b or 0 if division by zero."""
    return a / b if b else 0.0


def build_bipartite_graph(usage):
    """Create bipartite graph where left nodes are class_i, right nodes are option_j."""
    n_classes, n_options = usage.shape
    G = nx.Graph()
    # Add class nodes with prefix 'c' and bipartite attribute 0
    for c in range(n_classes):
        G.add_node(f"c{c}", bipartite=0)
    # Add option nodes with prefix 'o' and bipartite attribute 1
    for o in range(n_options):
        G.add_node(f"o{o}", bipartite=1)
    # Add edges where usage == 1
    rows, cols = np.where(usage == 1)
    for c, o in zip(rows, cols):
        G.add_edge(f"c{c}", f"o{o}")
    return G


def build_conflict_graph(usage):
    """Graph over classes: edge if two classes share at least one option."""
    n_classes, _ = usage.shape
    G = nx.Graph()
    G.add_nodes_from(range(n_classes))
    for i in range(n_classes):
        for j in range(i + 1, n_classes):
            if np.any(np.logical_and(usage[i], usage[j])):
                G.add_edge(i, j)
    return G


def main():
    # Get instance data using helper
    instance = input_data()

    # Extract arrays
    n_cars = int(instance.get("n_cars", 0))
    n_classes = int(instance.get("n_classes", 0))
    n_options = int(instance.get("n_options", 0))

    quantity = np.array(instance.get("quantity", []), dtype=int)
    maxcars = np.array(instance.get("maxcars", []), dtype=int)
    blksize_delta = np.array(instance.get("blksize_delta", []), dtype=int)
    usage_matrix = np.array(instance.get("usage", []), dtype=int)

    # Derived arrays
    window_size = maxcars + blksize_delta
    tightness = np.array([safe_div(m, w) for m, w in zip(maxcars, window_size)])

    # Basic stats
    avg_quantity = np.mean(quantity) if quantity.size else 0.0
    std_quantity = np.std(quantity, ddof=0) if quantity.size else 0.0
    min_quantity = int(np.min(quantity)) if quantity.size else 0
    max_quantity = int(np.max(quantity)) if quantity.size else 0
    entropy_quantity = shannon_entropy(quantity / quantity.sum()) if quantity.sum() else 0.0

    avg_maxcars = np.mean(maxcars) if maxcars.size else 0.0
    std_maxcars = np.std(maxcars, ddof=0) if maxcars.size else 0.0
    min_maxcars = int(np.min(maxcars)) if maxcars.size else 0
    max_maxcars = int(np.max(maxcars)) if maxcars.size else 0

    avg_blksize_delta = np.mean(blksize_delta) if blksize_delta.size else 0.0
    std_blksize_delta = np.std(blksize_delta, ddof=0) if blksize_delta.size else 0.0

    avg_window = np.mean(window_size) if window_size.size else 0.0
    std_window = np.std(window_size, ddof=0) if window_size.size else 0.0

    # Usage statistics
    total_possible_edges = n_classes * n_options
    total_edges = int(np.sum(usage_matrix))
    usage_density = safe_div(total_edges, total_possible_edges)

    options_per_class = np.sum(usage_matrix, axis=1) if usage_matrix.size else np.array([])
    avg_options_per_class = np.mean(options_per_class) if options_per_class.size else 0.0
    std_options_per_class = np.std(options_per_class, ddof=0) if options_per_class.size else 0.0
    min_options_per_class = int(np.min(options_per_class)) if options_per_class.size else 0
    max_options_per_class = int(np.max(options_per_class)) if options_per_class.size else 0

    classes_per_option = np.sum(usage_matrix, axis=0) if usage_matrix.size else np.array([])
    avg_classes_per_option = np.mean(classes_per_option) if classes_per_option.size else 0.0
    std_classes_per_option = np.std(classes_per_option, ddof=0) if classes_per_option.size else 0.0
    min_classes_per_option = int(np.min(classes_per_option)) if classes_per_option.size else 0
    max_classes_per_option = int(np.max(classes_per_option)) if classes_per_option.size else 0

    # Graph analyses
    bipartite_G = build_bipartite_graph(usage_matrix)
    bipartite_density = safe_div(total_edges, total_possible_edges)
    try:
        bipartite_assort = nx.degree_pearson_correlation_coefficient(bipartite_G)
    except Exception:
        bipartite_assort = 0.0

    conflict_G = build_conflict_graph(usage_matrix)
    conflict_nodes = conflict_G.number_of_nodes()
    conflict_edges = conflict_G.number_of_edges()
    possible_conflict_edges = safe_div(conflict_nodes * (conflict_nodes - 1), 2)
    conflict_density = safe_div(conflict_edges, possible_conflict_edges)
    conflict_avg_degree = safe_div(2 * conflict_edges, conflict_nodes)
    try:
        conflict_clustering = nx.average_clustering(conflict_G) if conflict_nodes > 0 else 0.0
    except Exception:
        conflict_clustering = 0.0
    try:
        conflict_assort = nx.degree_pearson_correlation_coefficient(conflict_G) if conflict_edges > 0 else 0.0
    except Exception:
        conflict_assort = 0.0

    # Tightness stats
    avg_tight = np.mean(tightness) if tightness.size else 0.0
    std_tight = np.std(tightness, ddof=0) if tightness.size else 0.0
    min_tight = float(np.min(tightness)) if tightness.size else 0.0
    max_tight = float(np.max(tightness)) if tightness.size else 0.0
    proportion_tight_options = safe_div(np.sum(tightness <= 0.5), tightness.size) if tightness.size else 0.0

    # Other ratios
    ratio_cars_classes = safe_div(n_cars, n_classes)
    ratio_options_classes = safe_div(n_options, n_classes)
    ratio_options_cars = safe_div(n_options, n_cars)

    proportion_heavy_classes = safe_div(np.sum(options_per_class > avg_options_per_class), n_classes) if n_classes else 0.0
    proportion_rare_options = safe_div(np.sum(classes_per_option <= 1), n_options) if n_options else 0.0

    # Variation and inequality metrics
    coeff_var_quantity = safe_div(std_quantity, avg_quantity)
    coeff_var_options_class = safe_div(std_options_per_class, avg_options_per_class)

    gini_q = gini_coefficient(quantity)
    gini_opt_per_class = gini_coefficient(options_per_class)

    max_demand_ratio = safe_div(max_quantity, n_cars)
    min_demand_ratio = safe_div(min_quantity, n_cars)

    # Skewness of quantity
    if quantity.size and std_quantity:
        skew_quantity = safe_div(np.mean(((quantity - avg_quantity) / std_quantity) ** 3), 1)
    else:
        skew_quantity = 0.0

    # Assemble results dictionary
    results = {
        "README": (
            "This script analyses Car Sequencing instances by transforming the class-option incidence matrix into "
            "two complementary graphs. The first is a bipartite graph linking each car class to the options it "
            "requires; its density, degree statistics and assortativity reflect how densely requirements couple "
            "classes to resources. From this graph we derive basic load features such as average options per class "
            "and classes per option. The second graph links classes that share at least one option, exposing direct "
            "competition for assembly-line capacity – properties such as edge density, clustering and degree "
            "assortativity capture constraint propagation potential. Global instance size is encoded through the number "
            "of cars, classes and options, while production demand heterogeneity is described via statistical moments, "
            "entropy, variation coefficients and Gini indices of the quantity vector. Window constraints are summarised "
            "by their average size and tightness (ratio maxcars/window) which directly influences search space pruning. "
            "Additional ratios, e.g. cars-to-classes or options-to-cars, plus proportions of heavy classes and rare "
            "options, quantify structural imbalance that often drives branching behaviour. Altogether, exactly fifty "
            "numeric characteristics are yielded, each named characteristic_1 … characteristic_50, providing a compact "
            "yet expressive fingerprint to guide automatic solver parameter selection."),
        "characteristic_1": n_cars,                          # total cars
        "characteristic_2": n_classes,                       # total classes
        "characteristic_3": n_options,                       # total options
        "characteristic_4": float(avg_quantity),             # avg quantity per class
        "characteristic_5": float(std_quantity),             # std quantity per class
        "characteristic_6": min_quantity,                    # min quantity
        "characteristic_7": max_quantity,                    # max quantity
        "characteristic_8": float(entropy_quantity),         # entropy of quantity distribution
        "characteristic_9": float(avg_maxcars),              # avg maxcars
        "characteristic_10": float(std_maxcars),             # std maxcars
        "characteristic_11": min_maxcars,                    # min maxcars
        "characteristic_12": max_maxcars,                    # max maxcars
        "characteristic_13": float(avg_blksize_delta),       # avg blksize_delta
        "characteristic_14": float(std_blksize_delta),       # std blksize_delta
        "characteristic_15": float(avg_window),              # avg window size
        "characteristic_16": float(std_window),              # std window size
        "characteristic_17": float(usage_density),           # usage density
        "characteristic_18": float(avg_options_per_class),   # avg options per class
        "characteristic_19": float(std_options_per_class),   # std options per class
        "characteristic_20": min_options_per_class,          # min options per class
        "characteristic_21": max_options_per_class,          # max options per class
        "characteristic_22": float(avg_classes_per_option),  # avg classes per option
        "characteristic_23": float(std_classes_per_option),  # std classes per option
        "characteristic_24": min_classes_per_option,         # min classes per option
        "characteristic_25": max_classes_per_option,         # max classes per option
        "characteristic_26": float(bipartite_density),       # bipartite density (same as usage density)
        "characteristic_27": float(bipartite_assort),        # bipartite degree assortativity
        "characteristic_28": conflict_nodes,                 # nodes in conflict graph
        "characteristic_29": conflict_edges,                 # edges in conflict graph
        "characteristic_30": float(conflict_density),        # density of conflict graph
        "characteristic_31": float(conflict_avg_degree),     # avg degree conflict graph
        "characteristic_32": float(conflict_clustering),     # clustering coefficient
        "characteristic_33": float(conflict_assort),         # assortativity conflict graph
        "characteristic_34": float(avg_tight),               # avg tightness
        "characteristic_35": float(std_tight),               # std tightness
        "characteristic_36": float(min_tight),               # min tightness
        "characteristic_37": float(max_tight),               # max tightness
        "characteristic_38": float(proportion_tight_options),# pct options tight <=0.5
        "characteristic_39": float(ratio_cars_classes),      # cars / classes
        "characteristic_40": float(ratio_options_classes),   # options / classes
        "characteristic_41": float(ratio_options_cars),      # options / cars
        "characteristic_42": float(proportion_heavy_classes),# classes w/ > avg options ratio
        "characteristic_43": float(proportion_rare_options), # options used by <=1 class
        "characteristic_44": float(coeff_var_quantity),      # CV of quantity
        "characteristic_45": float(coeff_var_options_class), # CV of options per class
        "characteristic_46": float(gini_q),                  # Gini of quantity
        "characteristic_47": float(gini_opt_per_class),      # Gini of options per class
        "characteristic_48": float(max_demand_ratio),        # max quantity / n_cars
        "characteristic_49": float(min_demand_ratio),        # min quantity / n_cars
        "characteristic_50": float(skew_quantity)            # skewness of quantity
    }

    # Output results
    output_results(results)


if __name__ == "__main__":
    main()
