import os
import re
import json
import argparse
from pathlib import Path

def dzn_to_json(dzn_path):
    json_dict = {}
    with open(dzn_path, 'r') as f:
        content = f.read()

    # Remove comments
    content = re.sub(r'%.*', '', content)

    # Find all assignments
    assignments = re.findall(r'(\w+)\s*=\s*([^;]+);', content)

    for key, value in assignments:
        value = value.strip()

        # Parse array
        if value.startswith('[') and value.endswith(']'):
            elements = [x.strip() for x in value[1:-1].split(',')]
            try:
                elements = [int(x) for x in elements]
            except ValueError:
                try:
                    elements = [float(x) for x in elements]
                except ValueError:
                    elements = [x.strip('"') for x in elements]
            json_dict[key] = elements

        # Parse set of integers
        elif value.startswith('{') and value.endswith('}'):
            elements = value[1:-1].split(',')
            try:
                json_dict[key] = [int(e.strip()) for e in elements]
            except ValueError:
                json_dict[key] = [e.strip('"') for e in elements]

        # Parse string
        elif value.startswith('"') and value.endswith('"'):
            json_dict[key] = value.strip('"')

        # Parse number
        else:
            try:
                json_dict[key] = int(value)
            except ValueError:
                try:
                    json_dict[key] = float(value)
                except ValueError:
                    json_dict[key] = value

    return json_dict

def batch_convert_dzn_to_json(input_dir, output_dir):
    input_dir = Path(input_dir)
    output_dir = Path(output_dir)
    output_dir.mkdir(parents=True, exist_ok=True)

    dzn_files = list(input_dir.glob('*.dzn'))
    for dzn_file in dzn_files:
        json_data = dzn_to_json(dzn_file)
        output_path = output_dir / (dzn_file.stem + '.json')
        with open(output_path, 'w') as json_file:
            json.dump(json_data, json_file, indent=4)

    print(f"Converted {len(dzn_files)} file(s) from '{input_dir}' to '{output_dir}'.")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Batch convert .dzn files to .json")
    parser.add_argument("input_dir", help="Directory containing .dzn files")
    parser.add_argument("output_dir", help="Directory to store .json files")

    args = parser.parse_args()
    batch_convert_dzn_to_json(args.input_dir, args.output_dir)

