# Import helper functions for input/output
# IMPORTANT: You MUST import these functions at the beginning of your script
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the problem data."""
    # Get the instance data using the helper function
    instance_data = input_data()

    # Initialize results dictionary with required structure
    readme_text = (
        "This instance was analyzed by constructing a complete undirected graph representation of the Vehicle Routing Problem (VRP) from the flattened distance matrix provided. "
        "The graph includes one depot node and N customer nodes, with weighted edges corresponding to travel distances. "
        "We then computed fundamental graph-based metrics, including density, degree distribution statistics, clustering coefficient, assortativity, connectivity, diameter, and shortest paths, to capture structural complexity and constraint tightness. "
        "Demand distribution statistics (mean, variance, skewness, and kurtosis) and demand-to-capacity ratios were derived to reflect load balancing and vehicle utilization characteristics. "
        "Distance metrics were examined both at the pairwise level and relative to the depot, capturing mean, variance, and extremal values. "
        "We further extracted minimum spanning tree properties to estimate underlying connectivity cost and span. "
        "Centrality measures such as degree and closeness centrality quantify node importance and potential bottlenecks. "
        "Finally, higher-order statistical moments for pairwise distances, demands, and degrees were calculated to assess distribution skewness and tail heaviness. "
        "By capturing these features, the resulting parameter vector reflects both local and global structural patterns, constraint tightness, resource utilization, and distributional complexity, enabling automated selector or tuner systems to map instance profiles to effective search heuristics and solver parameter configurations."
    )
    results = {"README": readme_text}

    # Pre-fill characteristic keys 1..50
    for i in range(1, 51):
        results[f"characteristic_{i}"] = 0.0

    try:
        # Extract basic data
        N = int(instance_data.get('N', 0))
        capacity = float(instance_data.get('Capacity', 0))
        demands = np.array(instance_data.get('Demand', []), dtype=float)
        dist_flat = np.array(instance_data.get('Distance', []), dtype=float)
        num_nodes = N + 1  # including depot

        # Build distance matrix
        D = dist_flat.reshape((num_nodes, num_nodes))

        # Pairwise distance statistics (i<j)
        triu_idx = np.triu_indices(num_nodes, k=1)
        dist_vals = D[triu_idx]
        mean_pairwise = dist_vals.mean() if dist_vals.size else 0.0
        std_pairwise = dist_vals.std() if dist_vals.size else 0.0
        min_pairwise = dist_vals.min() if dist_vals.size else 0.0
        max_pairwise = dist_vals.max() if dist_vals.size else 0.0
        # skewness and kurtosis for pairwise distances
        if dist_vals.size:
            skew_pair = ((dist_vals - mean_pairwise)**3).mean() / (std_pairwise**3) if std_pairwise else 0.0
            kurt_pair = ((dist_vals - mean_pairwise)**4).mean() / (std_pairwise**4) - 3.0 if std_pairwise else 0.0
        else:
            skew_pair = 0.0
            kurt_pair = 0.0

        # Create graph
        G = nx.Graph()
        G.add_nodes_from(range(num_nodes))
        for u, v in zip(*triu_idx):
            G.add_edge(u, v, weight=float(D[u, v]))

        # Graph metrics
        density = nx.density(G)
        degrees = np.array([deg for _, deg in G.degree()])
        avg_degree = degrees.mean() if degrees.size else 0.0
        std_degree = degrees.std() if degrees.size else 0.0
        # skewness and kurtosis for degrees
        if degrees.size:
            skew_degree = ((degrees - avg_degree)**3).mean() / (std_degree**3) if std_degree else 0.0
            kurt_degree = ((degrees - avg_degree)**4).mean() / (std_degree**4) - 3.0 if std_degree else 0.0
        else:
            skew_degree = 0.0
            kurt_degree = 0.0
        clustering_coeff = nx.average_clustering(G)
        assortativity = nx.degree_pearson_correlation_coefficient(G)
        n_components = nx.number_connected_components(G)
        is_conn = 1 if nx.is_connected(G) else 0
        try:
            diameter = nx.diameter(G)
        except nx.NetworkXError:
            diameter = -1
        try:
            avg_shortest = nx.average_shortest_path_length(G, weight='weight')
        except Exception:
            avg_shortest = -1

        # Demand statistics
        total_demand = demands.sum() if demands.size else 0.0
        avg_demand = demands.mean() if demands.size else 0.0
        std_demand = demands.std() if demands.size else 0.0
        min_demand = demands.min() if demands.size else 0.0
        max_demand = demands.max() if demands.size else 0.0
        if demands.size:
            skew_demand = ((demands - avg_demand)**3).mean() / (std_demand**3) if std_demand else 0.0
            kurt_demand = ((demands - avg_demand)**4).mean() / (std_demand**4) - 3.0 if std_demand else 0.0
        else:
            skew_demand = 0.0
            kurt_demand = 0.0
        # Demand-capacity ratios
        ratio_total = total_demand / (capacity * N) if capacity and N else 0.0
        ratio_avg = (demands / capacity).mean() if capacity and demands.size else 0.0
        ratio_max = (demands / capacity).max() if capacity and demands.size else 0.0

        # Edge weight statistics
        edge_weights = np.array([d['weight'] for u, v, d in G.edges(data=True)])
        num_edges = edge_weights.size
        mean_edge = edge_weights.mean() if num_edges else 0.0
        std_edge = edge_weights.std() if num_edges else 0.0
        min_edge = edge_weights.min() if num_edges else 0.0
        max_edge = edge_weights.max() if num_edges else 0.0

        # Depot distances
        depot_dists = D[0, 1:]
        mean_dep = depot_dists.mean() if depot_dists.size else 0.0
        std_dep = depot_dists.std() if depot_dists.size else 0.0
        min_dep = depot_dists.min() if depot_dists.size else 0.0
        max_dep = depot_dists.max() if depot_dists.size else 0.0

        # Minimum spanning tree
        mst = nx.minimum_spanning_tree(G, weight='weight')
        mst_w = np.array([d for u, v, d in mst.edges(data='weight')], dtype=float)
        mst_total = mst_w.sum() if mst_w.size else 0.0
        mst_mean = mst_w.mean() if mst_w.size else 0.0
        mst_std = mst_w.std() if mst_w.size else 0.0
        mst_min = mst_w.min() if mst_w.size else 0.0
        mst_max = mst_w.max() if mst_w.size else 0.0

        # Centrality measures
        deg_cent = np.array(list(nx.degree_centrality(G).values()), dtype=float)
        depot_deg_cent = deg_cent[0] if deg_cent.size else 0.0
        avg_deg_cent = deg_cent.mean() if deg_cent.size else 0.0
        std_deg_cent = deg_cent.std() if deg_cent.size else 0.0
        min_deg_cent = deg_cent.min() if deg_cent.size else 0.0
        max_deg_cent = deg_cent.max() if deg_cent.size else 0.0
        # Closeness centrality
        clos_cent = np.array(list(nx.closeness_centrality(G).values()), dtype=float)
        avg_clos = clos_cent.mean() if clos_cent.size else 0.0

        # Populate results
        results.update({
            'characteristic_1': N,
            'characteristic_2': num_nodes,
            'characteristic_3': capacity,
            'characteristic_4': total_demand,
            'characteristic_5': avg_demand,
            'characteristic_6': std_demand,
            'characteristic_7': min_demand,
            'characteristic_8': max_demand,
            'characteristic_9': ratio_total,
            'characteristic_10': ratio_avg,
            'characteristic_11': ratio_max,
            'characteristic_12': density,
            'characteristic_13': avg_degree,
            'characteristic_14': std_degree,
            'characteristic_15': clustering_coeff,
            'characteristic_16': assortativity,
            'characteristic_17': n_components,
            'characteristic_18': is_conn,
            'characteristic_19': diameter,
            'characteristic_20': avg_shortest,
            'characteristic_21': num_edges,
            'characteristic_22': mean_edge,
            'characteristic_23': std_edge,
            'characteristic_24': min_edge,
            'characteristic_25': max_edge,
            'characteristic_26': mean_dep,
            'characteristic_27': std_dep,
            'characteristic_28': min_dep,
            'characteristic_29': max_dep,
            'characteristic_30': mean_pairwise,
            'characteristic_31': std_pairwise,
            'characteristic_32': min_pairwise,
            'characteristic_33': max_pairwise,
            'characteristic_34': mst_total,
            'characteristic_35': mst_mean,
            'characteristic_36': mst_std,
            'characteristic_37': mst_min,
            'characteristic_38': mst_max,
            'characteristic_39': depot_deg_cent,
            'characteristic_40': avg_deg_cent,
            'characteristic_41': std_deg_cent,
            'characteristic_42': min_deg_cent,
            'characteristic_43': max_deg_cent,
            'characteristic_44': skew_pair,
            'characteristic_45': kurt_pair,
            'characteristic_46': skew_demand,
            'characteristic_47': kurt_demand,
            'characteristic_48': skew_degree,
            'characteristic_49': kurt_degree,
            'characteristic_50': avg_clos
        })

    except Exception as e:
        results['error'] = str(e)

    # Return the results using the helper function
    output_results(results)


if __name__ == "__main__":
    main()
