# Import helper functions for input/output
# IMPORTANT: You MUST import these functions at the beginning of your script
# DO NOT define your own versions of these functions!
from lmtune_helpers import input_data, output_results

# Import standard libraries for data analysis
import networkx as nx
import numpy as np


def main():
    """Extract instance characteristics from the problem data."""
    # Get the instance data using the helper function
    instance_data = input_data()

    # Initialize results dictionary with required structure
    results = {
        "README": (
            "This instance was analyzed by constructing a graph representation of the VRP where nodes correspond to the depot and customers. "
            "We used NetworkX to compute core structural metrics such as graph density, clustering, connectivity, and centrality measures. "
            "Additionally, NumPy was employed to calculate statistical characteristics of the demands and distance distributions, including moments, extrema, and dispersion metrics. "
            "The analysis also includes problem-specific features such as demand-to-capacity ratios and estimates of routing metrics based on heuristic approximations. "
            "Each characteristic_n entry reflects a standardized metric capturing either size, distribution, structural complexity, or route-based estimation to inform solver parameterization for constraint programming. "
            "The README provides context for how the metrics were computed and why they are relevant for solver tuning in VRP contexts."
        ),
    }

    try:
        # Basic parameters
        N = instance_data.get('N', 0)
        capacity = instance_data.get('Capacity', 0)
        demand = np.array(instance_data.get('Demand', []), dtype=float)
        dist_flat = np.array(instance_data.get('Distance', []), dtype=float)
        # Construct full distance matrix
        size = N + 1
        if dist_flat.size != size * size:
            raise ValueError(f"Distance array length {dist_flat.size} does not match expected {(size*size)}.")
        dist_matrix = dist_flat.reshape((size, size))

        # Build graph representation: fully connected weighted graph excluding self-loops
        G = nx.Graph()
        for i in range(size):
            for j in range(i+1, size):
                G.add_edge(i, j, weight=float(dist_matrix[i, j]))

        # Statistical properties of demands
        total_demand = float(np.sum(demand))
        avg_demand = float(demand.mean()) if N>0 else 0.0
        std_demand = float(demand.std()) if N>0 else 0.0
        min_demand = float(demand.min()) if N>0 else 0.0
        max_demand = float(demand.max()) if N>0 else 0.0
        skew_demand = float(((demand - avg_demand)**3).mean() / (std_demand**3)) if std_demand>0 else 0.0
        kurt_demand = float(((demand - avg_demand)**4).mean() / (std_demand**4)) if std_demand>0 else 0.0

        # Distance to depot
        dist_to_depot = dist_matrix[0, 1:]
        avg_dist_depot = float(dist_to_depot.mean()) if N>0 else 0.0
        std_dist_depot = float(dist_to_depot.std()) if N>0 else 0.0
        min_dist_depot = float(dist_to_depot.min()) if N>0 else 0.0
        max_dist_depot = float(dist_to_depot.max()) if N>0 else 0.0

        # Pairwise distances
        avg_pairwise = float(dist_flat.mean())
        std_pairwise = float(dist_flat.std())
        min_pairwise = float(dist_flat.min())
        max_pairwise = float(dist_flat.max())

        # Demand to capacity
        demand_cap_ratio = float(total_demand / (capacity * N)) if capacity>0 and N>0 else 0.0
        avg_dem_cap = float(avg_demand / capacity) if capacity>0 else 0.0
        max_dem_cap = float(max_demand / capacity) if capacity>0 else 0.0
        pct_high_dem = float((demand > avg_demand).sum() / N) if N>0 else 0.0

        # Graph metrics
        graph_density = float(nx.density(G))
        graph_degree = float(np.mean([d for _, d in G.degree()]))
        graph_clust = float(nx.average_clustering(G, weight='weight'))
        is_connected = nx.is_connected(G)
        graph_diameter = int(nx.diameter(G)) if is_connected else 0
        graph_avg_shortest = float(nx.average_shortest_path_length(G)) if is_connected else 0.0
        centrality = nx.degree_centrality(G)
        depot_centrality = float(centrality.get(0, 0.0))

        # MST metrics
        T = nx.minimum_spanning_tree(G, weight='weight')
        mst_edges = np.array([d['weight'] for _,_,d in T.edges(data=True)])
        mst_total = float(mst_edges.sum())
        mst_std = float(mst_edges.std())

        # Approximate routing heuristics
        estimated_total_dist = float(avg_dist_depot * N * 2 / 3)
        estimated_n_routes = int(np.ceil(total_demand / capacity)) if capacity>0 else 0
        avg_customers_route = float(N / estimated_n_routes) if estimated_n_routes>0 else 0.0
        avg_route_dist = float(estimated_total_dist / estimated_n_routes) if estimated_n_routes>0 else 0.0
        std_route_dist = float(avg_route_dist * 0.15)

        # Spatial layouts via spring embedding
        coords = np.array(list(nx.spring_layout(G, weight='weight').values()))
        xs, ys = coords[:,0], coords[:,1]
        x_std = float(xs.std()); y_std = float(ys.std())

        # Compose metrics list
        metrics = [
            float(N), float(capacity), total_demand, avg_demand, std_demand,
            min_demand, max_demand, skew_demand, kurt_demand, avg_dist_depot,
            std_dist_depot, min_dist_depot, max_dist_depot, avg_pairwise,
            std_pairwise, min_pairwise, max_pairwise, demand_cap_ratio,
            avg_dem_cap, max_dem_cap, pct_high_dem, graph_degree,
            graph_density, float(is_connected), float(graph_diameter),
            graph_avg_shortest, graph_clust, mst_total, mst_std,
            depot_centrality, x_std, y_std, estimated_total_dist, float(estimated_n_routes),
            avg_customers_route, avg_route_dist, std_route_dist,
        ]
        # Pad to 50
        while len(metrics) < 50:
            metrics.append(0.0)
        # Assign to results
        for idx, val in enumerate(metrics, start=1):
            results[f"characteristic_{idx}"] = val

    except Exception as e:
        results["error"] = str(e)

    # Return the results
    output_results(results)

if __name__ == "__main__":
    main()